<?php
/**
 * Performance Data Sync Script
 * Runs as cron job to sync ad performance data
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/functions.php';

// Get all active users with API credentials
$db = getDB();
$stmt = $db->prepare("
    SELECT DISTINCT u.id, u.email, ac.service_name, ac.credential_value
    FROM users u
    JOIN api_credentials ac ON u.id = ac.user_id
    WHERE ac.status = 'active' AND u.status = 'active'
    ORDER BY u.id
");
$stmt->execute();
$users = $stmt->fetchAll();

$syncResults = [];

foreach ($users as $user) {
    $userId = $user['id'];
    $serviceName = $user['service_name'];
    
    try {
        if ($serviceName === 'google_ads') {
            require_once __DIR__ . '/google_ads_enhanced.php';
            $googleAds = new GoogleAdsAPI($userId);
            $result = $googleAds->syncAdPerformance($userId);
            $syncResults[] = [
                'user_id' => $userId,
                'service' => 'google_ads',
                'success' => isset($result['success']) ? $result['success'] : false,
                'message' => $result['message'] ?? 'Unknown error'
            ];
        } elseif ($serviceName === 'facebook_ads') {
            require_once __DIR__ . '/facebook_ads_enhanced.php';
            $facebookAds = new FacebookAdsAPI($userId);
            $result = $facebookAds->syncAdPerformance($userId);
            $syncResults[] = [
                'user_id' => $userId,
                'service' => 'facebook_ads',
                'success' => isset($result['success']) ? $result['success'] : false,
                'message' => $result['message'] ?? 'Unknown error'
            ];
        }
    } catch (Exception $e) {
        error_log("Sync error for user $userId, service $serviceName: " . $e->getMessage());
        $syncResults[] = [
            'user_id' => $userId,
            'service' => $serviceName,
            'success' => false,
            'message' => $e->getMessage()
        ];
    }
}

// Log sync results
$logMessage = "Performance sync completed at " . date('Y-m-d H:i:s') . "\n";
$logMessage .= "Total users processed: " . count($users) . "\n";
$logMessage .= "Results: " . json_encode($syncResults) . "\n";

file_put_contents(__DIR__ . '/../logs/sync_performance.log', $logMessage, FILE_APPEND);

echo "Performance sync completed. Check logs for details.\n";
?>
