<?php
/**
 * Scheduled Email Sender
 * Runs as cron job to send scheduled email campaigns
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/functions.php';

// Get scheduled email campaigns
$db = getDB();
$stmt = $db->prepare("
    SELECT 
        ec.id,
        ec.user_id,
        ec.campaign_name,
        ec.subject,
        ec.html_content,
        ec.text_content,
        ec.recipient_count,
        ec.scheduled_at,
        u.email as user_email,
        u.first_name,
        u.last_name
    FROM email_campaigns ec
    JOIN users u ON ec.user_id = u.id
    WHERE ec.status = 'scheduled' 
    AND ec.scheduled_at <= NOW()
    AND ec.scheduled_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
    ORDER BY ec.scheduled_at ASC
");
$stmt->execute();
$scheduledCampaigns = $stmt->fetchAll();

$sendResults = [];

foreach ($scheduledCampaigns as $campaign) {
    try {
        // Get user's email service preference
        $stmt = $db->prepare("
            SELECT service_name, credential_value 
            FROM api_credentials 
            WHERE user_id = ? AND service_name IN ('sendgrid', 'mailchimp') AND status = 'active'
            LIMIT 1
        ");
        $stmt->execute([$campaign['user_id']]);
        $emailService = $stmt->fetch();
        
        if (!$emailService) {
            $sendResults[] = [
                'campaign_id' => $campaign['id'],
                'success' => false,
                'message' => 'No email service configured'
            ];
            continue;
        }
        
        // Initialize email service
        require_once __DIR__ . '/email_service_enhanced.php';
        $emailServiceObj = new EmailServiceEnhanced($emailService['service_name'], $campaign['user_id']);
        
        // Load recipients from CSV import table if available, fallback to user
        $to = [];
        try {
            $db->exec("CREATE TABLE IF NOT EXISTS email_campaign_recipients (
                id INT AUTO_INCREMENT PRIMARY KEY,
                campaign_id INT NOT NULL,
                email VARCHAR(255) NOT NULL,
                name VARCHAR(255) NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                UNIQUE KEY uniq_campaign_email (campaign_id, email),
                INDEX idx_campaign_id (campaign_id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");
            $rstmt = $db->prepare("SELECT email, name FROM email_campaign_recipients WHERE campaign_id = ?");
            $rstmt->execute([$campaign['id']]);
            $rows = $rstmt->fetchAll();
            foreach ($rows as $row) {
                $to[] = ['email' => $row['email'], 'name' => $row['name'] ?? ''];
            }
        } catch (Exception $e) {
            // ignore, fallback below
        }

        if (empty($to)) {
            $to[] = [
                'email' => $campaign['user_email'],
                'name' => $campaign['first_name'] . ' ' . $campaign['last_name']
            ];
        }

        // Prepare campaign data
        $campaignData = [
            'campaign_name' => $campaign['campaign_name'],
            'subject' => $campaign['subject'],
            'html_content' => $campaign['html_content'],
            'text_content' => $campaign['text_content'],
            'to' => $to,
            'campaign_id' => $campaign['id'],
            'user_id' => $campaign['user_id']
        ];
        
        // Send email
        $result = $emailServiceObj->sendEmailCampaign($campaignData);
        
        if (isset($result['success']) && $result['success']) {
            // Update campaign status
            $stmt = $db->prepare("
                UPDATE email_campaigns 
                SET status = 'sent', sent_at = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$campaign['id']]);
            
            // Sync campaign data
            $emailServiceObj->syncEmailCampaign($campaign['user_id'], $campaignData);
        }
        
        $sendResults[] = [
            'campaign_id' => $campaign['id'],
            'success' => isset($result['success']) ? $result['success'] : false,
            'message' => $result['message'] ?? 'Unknown error'
        ];
        
    } catch (Exception $e) {
        error_log("Email send error for campaign {$campaign['id']}: " . $e->getMessage());
        $sendResults[] = [
            'campaign_id' => $campaign['id'],
            'success' => false,
            'message' => $e->getMessage()
        ];
    }
}

// Log send results
$logMessage = "Scheduled email send completed at " . date('Y-m-d H:i:s') . "\n";
$logMessage .= "Campaigns processed: " . count($scheduledCampaigns) . "\n";
$logMessage .= "Results: " . json_encode($sendResults) . "\n";

file_put_contents(__DIR__ . '/../logs/send_emails.log', $logMessage, FILE_APPEND);

echo "Scheduled email send completed. Check logs for details.\n";
?>
