<?php
/**
 * Enhanced Reporting System
 * Provides comprehensive reporting with PDF/CSV export capabilities
 */

require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

class ReportingSystem {
    private $db;
    
    public function __construct() {
        $this->db = getDB();
    }
    
    /**
     * Generate comprehensive marketing report
     */
    public function generateMarketingReport($userId, $startDate, $endDate, $format = 'json') {
        try {
            $reportData = [
                'period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate
                ],
                'summary' => $this->getSummaryMetrics($userId, $startDate, $endDate),
                'campaigns' => $this->getCampaignData($userId, $startDate, $endDate),
                'ad_performance' => $this->getAdPerformanceData($userId, $startDate, $endDate),
                'email_campaigns' => $this->getEmailCampaignData($userId, $startDate, $endDate),
                'discount_codes' => $this->getDiscountCodeData($userId, $startDate, $endDate),
                'user_metrics' => $this->getUserMetricsData($userId, $startDate, $endDate),
                'direct_mail' => $this->getDirectMailData($userId, $startDate, $endDate)
            ];
            
            if ($format === 'pdf') {
                return $this->generatePDFReport($reportData);
            } elseif ($format === 'csv') {
                return $this->generateCSVReport($reportData);
            }
            
            return $reportData;
            
        } catch (Exception $e) {
            error_log("Reporting Error: " . $e->getMessage());
            return ['error' => 'Failed to generate report: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get summary metrics for the period
     */
    private function getSummaryMetrics($userId, $startDate, $endDate) {
        // Total ad spend
        $stmt = $this->db->prepare("
            SELECT 
                COALESCE(SUM(cost), 0) as total_ad_spend,
                COUNT(DISTINCT campaign_id) as total_campaigns,
                COALESCE(SUM(impressions), 0) as total_impressions,
                COALESCE(SUM(clicks), 0) as total_clicks,
                COALESCE(SUM(conversions), 0) as total_conversions
            FROM ad_performance 
            WHERE user_id = ? AND date BETWEEN ? AND ?
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        $adMetrics = $stmt->fetch();
        
        // Email metrics
        $stmt = $this->db->prepare("
            SELECT 
                COALESCE(SUM(recipient_count), 0) as total_emails_sent,
                COUNT(DISTINCT id) as total_email_campaigns
            FROM email_campaigns 
            WHERE user_id = ? AND sent_at BETWEEN ? AND ?
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        $emailMetrics = $stmt->fetch();
        
        // Discount code usage
        $stmt = $this->db->prepare("
            SELECT 
                COUNT(DISTINCT id) as total_discount_codes,
                COALESCE(SUM(usage_count), 0) as total_redemptions
            FROM discount_codes 
            WHERE user_id = ? AND created_at BETWEEN ? AND ?
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        $discountMetrics = $stmt->fetch();
        
        // User metrics
        $stmt = $this->db->prepare("
            SELECT 
                COUNT(DISTINCT session_id) as unique_visitors,
                COUNT(*) as total_page_views,
                COALESCE(SUM(CASE WHEN conversion_event IS NOT NULL THEN 1 ELSE 0 END), 0) as conversions
            FROM user_metrics 
            WHERE user_id = ? AND created_at BETWEEN ? AND ?
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        $userMetrics = $stmt->fetch();
        
        // Calculate ROI and other derived metrics
        $totalAdSpend = $adMetrics['total_ad_spend'];
        $totalConversions = $adMetrics['total_conversions'] + $userMetrics['conversions'];
        $totalClicks = $adMetrics['total_clicks'];
        $totalImpressions = $adMetrics['total_impressions'];
        
        $ctr = $totalImpressions > 0 ? ($totalClicks / $totalImpressions) * 100 : 0;
        $conversionRate = $totalClicks > 0 ? ($totalConversions / $totalClicks) * 100 : 0;
        $cpc = $totalClicks > 0 ? $totalAdSpend / $totalClicks : 0;
        
        return [
            'ad_spend' => $totalAdSpend,
            'total_campaigns' => $adMetrics['total_campaigns'],
            'total_impressions' => $totalImpressions,
            'total_clicks' => $totalClicks,
            'total_conversions' => $totalConversions,
            'ctr' => round($ctr, 2),
            'conversion_rate' => round($conversionRate, 2),
            'cpc' => round($cpc, 2),
            'emails_sent' => $emailMetrics['total_emails_sent'],
            'email_campaigns' => $emailMetrics['total_email_campaigns'],
            'discount_codes' => $discountMetrics['total_discount_codes'],
            'discount_redemptions' => $discountMetrics['total_redemptions'],
            'unique_visitors' => $userMetrics['unique_visitors'],
            'page_views' => $userMetrics['total_page_views']
        ];
    }
    
    /**
     * Get campaign data
     */
    private function getCampaignData($userId, $startDate, $endDate) {
        $stmt = $this->db->prepare("
            SELECT 
                c.id,
                c.campaign_name,
                c.campaign_type,
                c.status,
                c.budget,
                c.start_date,
                c.end_date,
                COALESCE(SUM(ap.cost), 0) as actual_spend,
                COALESCE(SUM(ap.impressions), 0) as impressions,
                COALESCE(SUM(ap.clicks), 0) as clicks,
                COALESCE(SUM(ap.conversions), 0) as conversions
            FROM campaigns c
            LEFT JOIN ad_performance ap ON c.id = ap.campaign_id AND ap.date BETWEEN ? AND ?
            WHERE c.user_id = ? AND c.created_at BETWEEN ? AND ?
            GROUP BY c.id
            ORDER BY c.created_at DESC
        ");
        $stmt->execute([$startDate, $endDate, $userId, $startDate, $endDate]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get ad performance data
     */
    private function getAdPerformanceData($userId, $startDate, $endDate) {
        $stmt = $this->db->prepare("
            SELECT 
                platform,
                campaign_id,
                campaign_name,
                SUM(impressions) as impressions,
                SUM(clicks) as clicks,
                SUM(cost) as cost,
                SUM(conversions) as conversions,
                AVG(ctr) as avg_ctr,
                AVG(cpc) as avg_cpc,
                AVG(cost_per_conversion) as avg_cost_per_conversion
            FROM ad_performance 
            WHERE user_id = ? AND date BETWEEN ? AND ?
            GROUP BY platform, campaign_id
            ORDER BY cost DESC
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get email campaign data
     */
    private function getEmailCampaignData($userId, $startDate, $endDate) {
        $stmt = $this->db->prepare("
            SELECT 
                ec.id,
                ec.campaign_name,
                ec.subject,
                ec.recipient_count,
                ec.sent_at,
                ec.status,
                COALESCE(ecs.delivered, 0) as delivered,
                COALESCE(ecs.opens, 0) as opens,
                COALESCE(ecs.clicks, 0) as clicks,
                COALESCE(ecs.bounces, 0) as bounces,
                COALESCE(ecs.open_rate, 0) as open_rate,
                COALESCE(ecs.click_rate, 0) as click_rate
            FROM email_campaigns ec
            LEFT JOIN email_campaign_stats ecs ON ec.id = ecs.campaign_id
            WHERE ec.user_id = ? AND ec.sent_at BETWEEN ? AND ?
            ORDER BY ec.sent_at DESC
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get discount code data
     */
    private function getDiscountCodeData($userId, $startDate, $endDate) {
        $stmt = $this->db->prepare("
            SELECT 
                code,
                discount_type,
                discount_value,
                usage_count,
                usage_limit,
                created_at,
                expires_at,
                status
            FROM discount_codes 
            WHERE user_id = ? AND created_at BETWEEN ? AND ?
            ORDER BY created_at DESC
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get user metrics data
     */
    private function getUserMetricsData($userId, $startDate, $endDate) {
        $stmt = $this->db->prepare("
            SELECT 
                DATE(created_at) as date,
                COUNT(DISTINCT session_id) as unique_visitors,
                COUNT(*) as page_views,
                COUNT(CASE WHEN conversion_event IS NOT NULL THEN 1 END) as conversions,
                COUNT(CASE WHEN utm_source IS NOT NULL THEN 1 END) as utm_traffic
            FROM user_metrics 
            WHERE user_id = ? AND created_at BETWEEN ? AND ?
            GROUP BY DATE(created_at)
            ORDER BY date DESC
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get direct mail data
     */
    private function getDirectMailData($userId, $startDate, $endDate) {
        $stmt = $this->db->prepare("
            SELECT 
                id,
                campaign_name,
                quantity,
                cost_per_piece,
                total_cost,
                scheduled_date,
                status,
                created_at
            FROM direct_mail_campaigns 
            WHERE user_id = ? AND created_at BETWEEN ? AND ?
            ORDER BY created_at DESC
        ");
        $stmt->execute([$userId, $startDate, $endDate]);
        return $stmt->fetchAll();
    }
    
    /**
     * Generate PDF report
     */
    private function generatePDFReport($reportData) {
        // This would require a PDF library like TCPDF or FPDF
        // For now, return a placeholder
        return [
            'success' => false,
            'message' => 'PDF generation requires PDF library installation',
            'suggestion' => 'Install TCPDF or FPDF library for PDF generation'
        ];
    }
    
    /**
     * Generate CSV report
     */
    private function generateCSVReport($reportData) {
        try {
            $csvData = [];
            
            // Summary section
            $csvData[] = ['Marketing Report Summary'];
            $csvData[] = ['Period', $reportData['period']['start_date'] . ' to ' . $reportData['period']['end_date']];
            $csvData[] = [];
            $csvData[] = ['Metric', 'Value'];
            $csvData[] = ['Total Ad Spend', '$' . number_format($reportData['summary']['ad_spend'], 2)];
            $csvData[] = ['Total Campaigns', $reportData['summary']['total_campaigns']];
            $csvData[] = ['Total Impressions', number_format($reportData['summary']['total_impressions'])];
            $csvData[] = ['Total Clicks', number_format($reportData['summary']['total_clicks'])];
            $csvData[] = ['Total Conversions', $reportData['summary']['total_conversions']];
            $csvData[] = ['CTR', $reportData['summary']['ctr'] . '%'];
            $csvData[] = ['Conversion Rate', $reportData['summary']['conversion_rate'] . '%'];
            $csvData[] = ['CPC', '$' . number_format($reportData['summary']['cpc'], 2)];
            $csvData[] = ['Emails Sent', number_format($reportData['summary']['emails_sent'])];
            $csvData[] = ['Unique Visitors', number_format($reportData['summary']['unique_visitors'])];
            $csvData[] = [];
            
            // Campaign performance
            $csvData[] = ['Campaign Performance'];
            $csvData[] = ['Campaign Name', 'Type', 'Status', 'Budget', 'Actual Spend', 'Impressions', 'Clicks', 'Conversions'];
            foreach ($reportData['campaigns'] as $campaign) {
                $csvData[] = [
                    $campaign['campaign_name'],
                    $campaign['campaign_type'],
                    $campaign['status'],
                    '$' . number_format($campaign['budget'], 2),
                    '$' . number_format($campaign['actual_spend'], 2),
                    number_format($campaign['impressions']),
                    number_format($campaign['clicks']),
                    $campaign['conversions']
                ];
            }
            $csvData[] = [];
            
            // Ad performance by platform
            $csvData[] = ['Ad Performance by Platform'];
            $csvData[] = ['Platform', 'Campaign', 'Impressions', 'Clicks', 'Cost', 'Conversions', 'CTR', 'CPC'];
            foreach ($reportData['ad_performance'] as $ad) {
                $csvData[] = [
                    $ad['platform'],
                    $ad['campaign_name'],
                    number_format($ad['impressions']),
                    number_format($ad['clicks']),
                    '$' . number_format($ad['cost'], 2),
                    $ad['conversions'],
                    number_format($ad['avg_ctr'], 2) . '%',
                    '$' . number_format($ad['avg_cpc'], 2)
                ];
            }
            
            // Convert to CSV string
            $csvString = '';
            foreach ($csvData as $row) {
                $csvString .= '"' . implode('","', $row) . '"' . "\n";
            }
            
            return [
                'success' => true,
                'data' => $csvString,
                'filename' => 'marketing_report_' . date('Y-m-d') . '.csv'
            ];
            
        } catch (Exception $e) {
            error_log("CSV Generation Error: " . $e->getMessage());
            return ['error' => 'Failed to generate CSV: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get real-time dashboard data
     */
    public function getDashboardData($userId) {
        try {
            $today = date('Y-m-d');
            $yesterday = date('Y-m-d', strtotime('-1 day'));
            $lastWeek = date('Y-m-d', strtotime('-7 days'));
            $lastMonth = date('Y-m-d', strtotime('-30 days'));
            
            return [
                'today' => $this->getSummaryMetrics($userId, $today, $today),
                'yesterday' => $this->getSummaryMetrics($userId, $yesterday, $yesterday),
                'last_week' => $this->getSummaryMetrics($userId, $lastWeek, $today),
                'last_month' => $this->getSummaryMetrics($userId, $lastMonth, $today),
                'recent_campaigns' => $this->getRecentCampaigns($userId),
                'top_performing_ads' => $this->getTopPerformingAds($userId),
                'conversion_trends' => $this->getConversionTrends($userId)
            ];
            
        } catch (Exception $e) {
            error_log("Dashboard Data Error: " . $e->getMessage());
            return ['error' => 'Failed to get dashboard data: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get recent campaigns
     */
    private function getRecentCampaigns($userId) {
        $stmt = $this->db->prepare("
            SELECT 
                campaign_name,
                campaign_type,
                status,
                created_at
            FROM campaigns 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT 5
        ");
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get top performing ads
     */
    private function getTopPerformingAds($userId) {
        $stmt = $this->db->prepare("
            SELECT 
                platform,
                campaign_name,
                SUM(conversions) as conversions,
                SUM(cost) as cost,
                AVG(ctr) as ctr
            FROM ad_performance 
            WHERE user_id = ? AND date >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            GROUP BY platform, campaign_name
            ORDER BY conversions DESC
            LIMIT 5
        ");
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get conversion trends
     */
    private function getConversionTrends($userId) {
        $stmt = $this->db->prepare("
            SELECT 
                DATE(created_at) as date,
                COUNT(CASE WHEN conversion_event IS NOT NULL THEN 1 END) as conversions
            FROM user_metrics 
            WHERE user_id = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY DATE(created_at)
            ORDER BY date ASC
        ");
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
}

// API endpoint for reporting operations
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    require_once __DIR__ . '/../includes/auth.php';
    requireClient();
    
    $userId = $_SESSION['user_id'];
    $reporting = new ReportingSystem();
    $action = $_POST['action'];
    
    header('Content-Type: application/json');
    
    switch ($action) {
        case 'generate_report':
            $startDate = $_POST['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
            $endDate = $_POST['end_date'] ?? date('Y-m-d');
            $format = $_POST['format'] ?? 'json';
            
            $result = $reporting->generateMarketingReport($userId, $startDate, $endDate, $format);
            echo json_encode($result);
            break;
            
        case 'get_dashboard_data':
            $result = $reporting->getDashboardData($userId);
            echo json_encode($result);
            break;
            
        case 'export_csv':
            $startDate = $_POST['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
            $endDate = $_POST['end_date'] ?? date('Y-m-d');
            
            $result = $reporting->generateMarketingReport($userId, $startDate, $endDate, 'csv');
            
            if (isset($result['success']) && $result['success']) {
                header('Content-Type: text/csv');
                header('Content-Disposition: attachment; filename="' . $result['filename'] . '"');
                echo $result['data'];
            } else {
                echo json_encode($result);
            }
            break;
            
        default:
            echo json_encode(['error' => 'Invalid action']);
    }
    exit;
}
?>
