<?php
/**
 * User Metrics Tracking API
 * Tracks visitor metrics from UTM parameters, conversions, and user behavior
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/webhook_helper.php';

// Get user_id from query parameter (client should pass their user_id when embedding tracking)
$userId = isset($_GET['user_id']) ? intval($_GET['user_id']) : null;

if (!$userId) {
    http_response_code(400);
    echo json_encode(['error' => 'user_id parameter required']);
    exit;
}

// Collect tracking data
$data = [
    'user_id' => $userId,
    'session_id' => $_GET['session_id'] ?? session_id() ?? bin2hex(random_bytes(16)),
    'page_url' => $_GET['url'] ?? $_SERVER['HTTP_REFERER'] ?? '',
    'referrer' => $_GET['ref'] ?? $_SERVER['HTTP_REFERER'] ?? '',
    'utm_source' => $_GET['utm_source'] ?? null,
    'utm_medium' => $_GET['utm_medium'] ?? null,
    'utm_campaign' => $_GET['utm_campaign'] ?? null,
    'utm_term' => $_GET['utm_term'] ?? null,
    'utm_content' => $_GET['utm_content'] ?? null,
    'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
    'device_type' => getDeviceType($_SERVER['HTTP_USER_AGENT'] ?? ''),
    'conversion_type' => $_GET['conversion_type'] ?? null,
    'conversion_value' => isset($_GET['conversion_value']) ? floatval($_GET['conversion_value']) : null,
    'discount_code_used' => $_GET['discount_code'] ?? null,
];

// Log the metric
try {
    $success = logUserMetric($userId, $data);
    
    if ($success) {
        // Handle discount code usage if provided
        if (!empty($data['discount_code_used']) && !empty($data['conversion_value'])) {
            // Validate and record discount code usage
            $validation = validateAndUseDiscountCode($data['discount_code_used'], $data['conversion_value']);
            if ($validation['valid']) {
                recordDiscountCodeUsage(
                    $validation['discount_id'],
                    $userId,
                    $_GET['order_id'] ?? null,
                    $data['conversion_value'],
                    $validation['discount_amount'],
                    $data['conversion_value']
                );
            }
        }
        
        // Trigger webhook for conversion tracking if this is a conversion
        // Set session user_id temporarily for webhook trigger (this is the tracking code owner)
        if (!empty($data['conversion_type'])) {
            $originalUserId = $_SESSION['user_id'] ?? null;
            $_SESSION['user_id'] = $userId;
            triggerWebhookEvent('conversion.tracked', [
                'user_id' => $userId,
                'conversion_type' => $data['conversion_type'],
                'conversion_value' => $data['conversion_value'],
                'discount_code_used' => $data['discount_code_used'],
                'utm_source' => $data['utm_source'],
                'utm_campaign' => $data['utm_campaign'],
                'timestamp' => date('c')
            ]);
            if ($originalUserId !== null) {
                $_SESSION['user_id'] = $originalUserId;
            } else {
                unset($_SESSION['user_id']);
            }
        }
        
        echo json_encode([
            'success' => true,
            'message' => 'Metric logged successfully',
            'session_id' => $data['session_id']
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to log metric']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

