<?php
/**
 * IMAP Automation Worker
 * Reads messages from configured mailbox and logs/processes them.
 * Requires PHP IMAP extension enabled.
 */

require_once __DIR__ . '/../config/email_config.php';
require_once __DIR__ . '/../includes/database.php';

if (!function_exists('imap_open')) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'IMAP extension not enabled']);
    exit;
}

$host = IMAP_HOST;
$port = (int)IMAP_PORT;
$encryption = IMAP_ENCRYPTION; // ssl|tls|''
$user = IMAP_USER;
$pass = IMAP_PASS;
$mailbox = IMAP_MAILBOX ?: 'INBOX';

if (!$host || !$user || !$pass) {
    echo json_encode(['success' => false, 'message' => 'IMAP not configured']);
    exit;
}

$flags = ($encryption === 'ssl') ? '/ssl' : (($encryption === 'tls') ? '/tls' : '');
$mailboxPath = sprintf('{%s:%d%s}%s', $host, $port ?: 993, $flags, $mailbox);

$inbox = @imap_open($mailboxPath, $user, $pass);
if (!$inbox) {
    echo json_encode(['success' => false, 'message' => 'Failed to open mailbox: ' . imap_last_error()]);
    exit;
}

$emails = imap_search($inbox, 'UNSEEN');
$processed = 0;

if ($emails) {
    rsort($emails);
    foreach ($emails as $emailNumber) {
        $overview = imap_fetch_overview($inbox, $emailNumber, 0);
        $message = imap_fetchbody($inbox, $emailNumber, 1.1);
        if (empty($message)) {
            $message = imap_fetchbody($inbox, $emailNumber, 1);
        }

        // Basic processing: store into email_inbox table if exists, else just mark seen
        try {
            $db = getDB();
            $db->exec("CREATE TABLE IF NOT EXISTS email_inbox (
                id INT AUTO_INCREMENT PRIMARY KEY,
                message_id VARCHAR(255) NULL,
                subject VARCHAR(512) NULL,
                from_email VARCHAR(255) NULL,
                date_received DATETIME NULL,
                body MEDIUMTEXT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

            $ov = $overview[0] ?? null;
            $subject = $ov ? (isset($ov->subject) ? imap_utf8($ov->subject) : '') : '';
            $from = $ov ? ($ov->from ?? '') : '';
            $date = $ov ? ($ov->date ?? '') : '';
            $msgId = $ov ? ($ov->message_id ?? null) : null;

            $stmt = $db->prepare("INSERT INTO email_inbox (message_id, subject, from_email, date_received, body) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([
                $msgId,
                $subject,
                $from,
                date('Y-m-d H:i:s', strtotime($date ?: 'now')),
                $message
            ]);
            $processed++;
        } catch (\Throwable $e) {
            error_log('IMAP processing error: ' . $e->getMessage());
        }

        // Mark as seen
        imap_setflag_full($inbox, (string)$emailNumber, "\\Seen");
    }
}

imap_close($inbox);

echo json_encode(['success' => true, 'processed' => $processed]);


