<?php
/**
 * Enhanced Google Ads API Integration
 * Requires: Google Ads API PHP Library
 * Installation: composer require googleads/google-ads-php
 */

require_once __DIR__ . '/../config/google_ads_config.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/credential_helper.php';

class GoogleAdsAPI {
    private $clientId;
    private $clientSecret;
    private $refreshToken;
    private $developerToken;
    private $apiVersion;
    private $customerId;
    
    public function __construct($userId = null) {
        $this->clientId = GOOGLE_ADS_CLIENT_ID;
        $this->clientSecret = GOOGLE_ADS_CLIENT_SECRET;
        $this->refreshToken = GOOGLE_ADS_REFRESH_TOKEN;
        $this->developerToken = GOOGLE_ADS_DEVELOPER_TOKEN;
        $this->apiVersion = GOOGLE_ADS_API_VERSION;
        
        // If user-specific credentials are provided, use those
        if ($userId) {
            $userCreds = getUserCredentials($userId, 'google_ads');
            if ($userCreds) {
                $this->clientId = $userCreds['client_id'] ?? $this->clientId;
                $this->clientSecret = $userCreds['client_secret'] ?? $this->clientSecret;
                $this->refreshToken = $userCreds['refresh_token'] ?? $this->refreshToken;
                $this->developerToken = $userCreds['developer_token'] ?? $this->developerToken;
                $this->customerId = $userCreds['customer_id'] ?? null;
            }
        }
    }
    
    /**
     * Get campaign performance data
     */
    public function getCampaignPerformance($customerId, $startDate, $endDate) {
        try {
            // Check if Google Ads API library is available
            if (!class_exists('Google\Ads\GoogleAds\V14\GoogleAdsServiceClient')) {
                // Fallback to mock data if library not installed
                return $this->getMockCampaignData($startDate, $endDate);
            }
            
            // Initialize Google Ads client
            $googleAdsClient = $this->initializeGoogleAdsClient();
            $googleAdsServiceClient = $googleAdsClient->getGoogleAdsServiceClient();
            
            // Build query for campaign performance
            $query = "
                SELECT 
                    campaign.id,
                    campaign.name,
                    campaign.status,
                    campaign.advertising_channel_type,
                    metrics.impressions,
                    metrics.clicks,
                    metrics.ctr,
                    metrics.average_cpc,
                    metrics.cost_micros,
                    metrics.conversions,
                    metrics.cost_per_conversion,
                    metrics.conversion_rate,
                    segments.date
                FROM campaign
                WHERE segments.date BETWEEN '$startDate' AND '$endDate'
                AND campaign.status IN ('ENABLED', 'PAUSED')
                ORDER BY segments.date DESC
            ";
            
            $response = $googleAdsServiceClient->search($customerId, $query);
            
            $campaigns = [];
            $summary = [
                'total_impressions' => 0,
                'total_clicks' => 0,
                'total_cost' => 0,
                'total_conversions' => 0,
                'average_ctr' => 0,
                'average_cpc' => 0,
                'average_conversion_rate' => 0
            ];
            
            foreach ($response->iterateAllElements() as $row) {
                $campaign = $row->getCampaign();
                $metrics = $row->getMetrics();
                $segments = $row->getSegments();
                
                $cost = $metrics->getCostMicros() / 1000000; // Convert from micros
                $cpc = $metrics->getAverageCpc() ? $metrics->getAverageCpc() / 1000000 : 0;
                $costPerConversion = $metrics->getCostPerConversion() ? $metrics->getCostPerConversion() / 1000000 : 0;
                
                $campaignData = [
                    'id' => $campaign->getId(),
                    'name' => $campaign->getName(),
                    'status' => $campaign->getStatus(),
                    'channel_type' => $campaign->getAdvertisingChannelType(),
                    'date' => $segments->getDate(),
                    'impressions' => $metrics->getImpressions(),
                    'clicks' => $metrics->getClicks(),
                    'ctr' => $metrics->getCtr(),
                    'average_cpc' => $cpc,
                    'cost' => $cost,
                    'conversions' => $metrics->getConversions(),
                    'cost_per_conversion' => $costPerConversion,
                    'conversion_rate' => $metrics->getConversionRate()
                ];
                
                $campaigns[] = $campaignData;
                
                // Update summary
                $summary['total_impressions'] += $metrics->getImpressions();
                $summary['total_clicks'] += $metrics->getClicks();
                $summary['total_cost'] += $cost;
                $summary['total_conversions'] += $metrics->getConversions();
            }
            
            // Calculate averages
            if ($summary['total_clicks'] > 0) {
                $summary['average_ctr'] = ($summary['total_clicks'] / $summary['total_impressions']) * 100;
                $summary['average_cpc'] = $summary['total_cost'] / $summary['total_clicks'];
            }
            if ($summary['total_clicks'] > 0) {
                $summary['average_conversion_rate'] = ($summary['total_conversions'] / $summary['total_clicks']) * 100;
            }
            
            return [
                'campaigns' => $campaigns,
                'summary' => $summary
            ];
            
        } catch (Exception $e) {
            error_log("Google Ads API Error: " . $e->getMessage());
            return $this->getMockCampaignData($startDate, $endDate);
        }
    }
    
    /**
     * Create a new Google Ads campaign
     */
    public function createCampaign($customerId, $campaignData) {
        try {
            if (!class_exists('Google\Ads\GoogleAds\V14\GoogleAdsServiceClient')) {
                return ['error' => 'Google Ads API library not installed'];
            }
            
            $googleAdsClient = $this->initializeGoogleAdsClient();
            $campaignServiceClient = $googleAdsClient->getCampaignServiceClient();
            
            // Create campaign operation
            $campaignOperation = new \Google\Ads\GoogleAds\V14\Services\CampaignOperation();
            $campaign = new \Google\Ads\GoogleAds\V14\Resources\Campaign();
            
            $campaign->setName($campaignData['name']);
            $campaign->setAdvertisingChannelType($campaignData['channel_type'] ?? 'SEARCH');
            $campaign->setStatus($campaignData['status'] ?? 'ENABLED');
            
            // Set budget
            $campaignBudget = new \Google\Ads\GoogleAds\V14\Resources\CampaignBudget();
            $campaignBudget->setName($campaignData['name'] . ' Budget');
            $campaignBudget->setAmountMicros($campaignData['budget'] * 1000000);
            $campaignBudget->setDeliveryMethod('STANDARD');
            
            $budgetOperation = new \Google\Ads\GoogleAds\V14\Services\CampaignBudgetOperation();
            $budgetOperation->setCreate($campaignBudget);
            
            // Execute budget creation
            $budgetResponse = $campaignServiceClient->mutateCampaignBudgets($customerId, [$budgetOperation]);
            $budgetResourceName = $budgetResponse->getResults()[0]->getResourceName();
            
            $campaign->setCampaignBudget($budgetResourceName);
            
            $campaignOperation->setCreate($campaign);
            
            // Execute campaign creation
            $response = $campaignServiceClient->mutateCampaigns($customerId, [$campaignOperation]);
            
            return [
                'success' => true,
                'campaign_id' => $response->getResults()[0]->getResourceName(),
                'message' => 'Campaign created successfully'
            ];
            
        } catch (Exception $e) {
            error_log("Google Ads Campaign Creation Error: " . $e->getMessage());
            return ['error' => 'Failed to create campaign: ' . $e->getMessage()];
        }
    }
    
    /**
     * Update campaign budget
     */
    public function updateCampaignBudget($customerId, $campaignId, $newBudget) {
        try {
            if (!class_exists('Google\Ads\GoogleAds\V14\GoogleAdsServiceClient')) {
                return ['error' => 'Google Ads API library not installed'];
            }
            
            $googleAdsClient = $this->initializeGoogleAdsClient();
            $campaignBudgetServiceClient = $googleAdsClient->getCampaignBudgetServiceClient();
            
            // Get current campaign budget
            $query = "
                SELECT campaign_budget.resource_name, campaign_budget.amount_micros
                FROM campaign_budget
                WHERE campaign.id = $campaignId
            ";
            
            $googleAdsServiceClient = $googleAdsClient->getGoogleAdsServiceClient();
            $response = $googleAdsServiceClient->search($customerId, $query);
            
            foreach ($response->iterateAllElements() as $row) {
                $budget = $row->getCampaignBudget();
                
                $budgetOperation = new \Google\Ads\GoogleAds\V14\Services\CampaignBudgetOperation();
                $budgetUpdate = new \Google\Ads\GoogleAds\V14\Resources\CampaignBudget();
                $budgetUpdate->setResourceName($budget->getResourceName());
                $budgetUpdate->setAmountMicros($newBudget * 1000000);
                
                $budgetOperation->setUpdate($budgetUpdate);
                
                $response = $campaignBudgetServiceClient->mutateCampaignBudgets($customerId, [$budgetOperation]);
                
                return [
                    'success' => true,
                    'message' => 'Budget updated successfully',
                    'new_budget' => $newBudget
                ];
            }
            
            return ['error' => 'Campaign budget not found'];
            
        } catch (Exception $e) {
            error_log("Google Ads Budget Update Error: " . $e->getMessage());
            return ['error' => 'Failed to update budget: ' . $e->getMessage()];
        }
    }
    
    /**
     * Sync ad performance data to database
     */
    public function syncAdPerformance($userId) {
        try {
            $db = getDB();
            $customerId = $this->customerId;
            
            if (!$customerId) {
                return ['error' => 'Customer ID not configured'];
            }
            
            // Get performance data for last 30 days
            $endDate = date('Y-m-d');
            $startDate = date('Y-m-d', strtotime('-30 days'));
            
            $performanceData = $this->getCampaignPerformance($customerId, $startDate, $endDate);
            
            if (isset($performanceData['error'])) {
                return $performanceData;
            }
            
            // Store performance data in database
            foreach ($performanceData['campaigns'] as $campaign) {
                $stmt = $db->prepare("
                    INSERT INTO ad_performance 
                    (user_id, platform, campaign_id, campaign_name, date, impressions, clicks, cost, conversions, ctr, cpc, cost_per_conversion)
                    VALUES (?, 'google_ads', ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE
                    impressions = VALUES(impressions),
                    clicks = VALUES(clicks),
                    cost = VALUES(cost),
                    conversions = VALUES(conversions),
                    ctr = VALUES(ctr),
                    cpc = VALUES(cpc),
                    cost_per_conversion = VALUES(cost_per_conversion),
                    updated_at = NOW()
                ");
                
                $stmt->execute([
                    $userId,
                    $campaign['id'],
                    $campaign['name'],
                    $campaign['date'],
                    $campaign['impressions'],
                    $campaign['clicks'],
                    $campaign['cost'],
                    $campaign['conversions'],
                    $campaign['ctr'],
                    $campaign['average_cpc'],
                    $campaign['cost_per_conversion']
                ]);
            }
            
            return [
                'success' => true,
                'message' => 'Performance data synced successfully',
                'campaigns_synced' => count($performanceData['campaigns'])
            ];
            
        } catch (Exception $e) {
            error_log("Google Ads Sync Error: " . $e->getMessage());
            return ['error' => 'Failed to sync performance data: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get budget alerts and monitoring
     */
    public function getBudgetAlerts($customerId) {
        try {
            if (!class_exists('Google\Ads\GoogleAds\V14\GoogleAdsServiceClient')) {
                return $this->getMockBudgetAlerts();
            }
            
            $googleAdsClient = $this->initializeGoogleAdsClient();
            $googleAdsServiceClient = $googleAdsClient->getGoogleAdsServiceClient();
            
            // Query for budget usage
            $query = "
                SELECT 
                    campaign.id,
                    campaign.name,
                    campaign_budget.amount_micros,
                    campaign_budget.period,
                    metrics.cost_micros,
                    campaign.status
                FROM campaign
                WHERE campaign.status IN ('ENABLED', 'PAUSED')
                AND campaign_budget.period = 'DAILY'
            ";
            
            $response = $googleAdsServiceClient->search($customerId, $query);
            
            $alerts = [];
            $currentDate = date('Y-m-d');
            
            foreach ($response->iterateAllElements() as $row) {
                $campaign = $row->getCampaign();
                $budget = $row->getCampaignBudget();
                $metrics = $row->getMetrics();
                
                $dailyBudget = $budget->getAmountMicros() / 1000000;
                $spentToday = $metrics->getCostMicros() / 1000000;
                $usagePercentage = $dailyBudget > 0 ? ($spentToday / $dailyBudget) * 100 : 0;
                
                $alertType = null;
                if ($usagePercentage >= 90) {
                    $alertType = 'budget_critical';
                } elseif ($usagePercentage >= 75) {
                    $alertType = 'budget_warning';
                } elseif ($usagePercentage >= 50) {
                    $alertType = 'budget_info';
                }
                
                if ($alertType) {
                    $alerts[] = [
                        'campaign_id' => $campaign->getId(),
                        'campaign_name' => $campaign->getName(),
                        'budget_used' => round($usagePercentage, 2),
                        'daily_budget' => $dailyBudget,
                        'spent_today' => $spentToday,
                        'alert_type' => $alertType,
                        'date' => $currentDate
                    ];
                }
            }
            
            return ['alerts' => $alerts];
            
        } catch (Exception $e) {
            error_log("Google Ads Budget Alerts Error: " . $e->getMessage());
            return $this->getMockBudgetAlerts();
        }
    }
    
    /**
     * Initialize Google Ads client
     */
    private function initializeGoogleAdsClient() {
        return new \Google\Ads\GoogleAds\V14\GoogleAdsClient([
            'developerToken' => $this->developerToken,
            'clientId' => $this->clientId,
            'clientSecret' => $this->clientSecret,
            'refreshToken' => $this->refreshToken
        ]);
    }
    
    /**
     * Get mock campaign data for testing
     */
    private function getMockCampaignData($startDate, $endDate) {
        return [
            'campaigns' => [
                [
                    'id' => '123456789',
                    'name' => 'Sample Search Campaign',
                    'status' => 'ENABLED',
                    'channel_type' => 'SEARCH',
                    'date' => $endDate,
                    'impressions' => 15000,
                    'clicks' => 450,
                    'ctr' => 3.0,
                    'average_cpc' => 1.25,
                    'cost' => 562.50,
                    'conversions' => 12,
                    'cost_per_conversion' => 46.88,
                    'conversion_rate' => 2.67
                ],
                [
                    'id' => '987654321',
                    'name' => 'Sample Display Campaign',
                    'status' => 'ENABLED',
                    'channel_type' => 'DISPLAY',
                    'date' => $endDate,
                    'impressions' => 25000,
                    'clicks' => 300,
                    'ctr' => 1.2,
                    'average_cpc' => 0.85,
                    'cost' => 255.00,
                    'conversions' => 8,
                    'cost_per_conversion' => 31.88,
                    'conversion_rate' => 2.67
                ]
            ],
            'summary' => [
                'total_impressions' => 40000,
                'total_clicks' => 750,
                'total_cost' => 817.50,
                'total_conversions' => 20,
                'average_ctr' => 1.875,
                'average_cpc' => 1.09,
                'average_conversion_rate' => 2.67
            ]
        ];
    }
    
    /**
     * Get mock budget alerts for testing
     */
    private function getMockBudgetAlerts() {
        return [
            'alerts' => [
                [
                    'campaign_id' => '123456789',
                    'campaign_name' => 'Sample Search Campaign',
                    'budget_used' => 85.5,
                    'daily_budget' => 100.00,
                    'spent_today' => 85.50,
                    'alert_type' => 'budget_warning',
                    'date' => date('Y-m-d')
                ],
                [
                    'campaign_id' => '987654321',
                    'campaign_name' => 'Sample Display Campaign',
                    'budget_used' => 95.2,
                    'daily_budget' => 50.00,
                    'spent_today' => 47.60,
                    'alert_type' => 'budget_critical',
                    'date' => date('Y-m-d')
                ]
            ]
        ];
    }
}

// API endpoint for Google Ads operations
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    require_once __DIR__ . '/../includes/auth.php';
    requireClient();
    
    $userId = $_SESSION['user_id'];
    $googleAds = new GoogleAdsAPI($userId);
    $action = $_POST['action'];
    
    header('Content-Type: application/json');
    
    switch ($action) {
        case 'sync_performance':
            $result = $googleAds->syncAdPerformance($userId);
            echo json_encode($result);
            break;
            
        case 'get_budget_alerts':
            $customerId = $_POST['customer_id'] ?? null;
            if (!$customerId) {
                echo json_encode(['error' => 'Customer ID required']);
                break;
            }
            $result = $googleAds->getBudgetAlerts($customerId);
            echo json_encode($result);
            break;
            
        case 'create_campaign':
            $customerId = $_POST['customer_id'] ?? null;
            $campaignData = [
                'name' => $_POST['campaign_name'] ?? '',
                'budget' => floatval($_POST['budget'] ?? 0),
                'channel_type' => $_POST['channel_type'] ?? 'SEARCH',
                'status' => $_POST['status'] ?? 'ENABLED'
            ];
            
            if (!$customerId || empty($campaignData['name']) || $campaignData['budget'] <= 0) {
                echo json_encode(['error' => 'Invalid campaign data']);
                break;
            }
            
            $result = $googleAds->createCampaign($customerId, $campaignData);
            echo json_encode($result);
            break;
            
        case 'update_budget':
            $customerId = $_POST['customer_id'] ?? null;
            $campaignId = $_POST['campaign_id'] ?? null;
            $newBudget = floatval($_POST['new_budget'] ?? 0);
            
            if (!$customerId || !$campaignId || $newBudget <= 0) {
                echo json_encode(['error' => 'Invalid budget update data']);
                break;
            }
            
            $result = $googleAds->updateCampaignBudget($customerId, $campaignId, $newBudget);
            echo json_encode($result);
            break;
            
        default:
            echo json_encode(['error' => 'Invalid action']);
    }
    exit;
}
?>
