<?php
/**
 * Daily Report Generator
 * Runs as cron job to generate daily reports for all active users
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/functions.php';

// Get all active users
$db = getDB();
$stmt = $db->prepare("
    SELECT id, email, first_name, last_name
    FROM users 
    WHERE status = 'active' AND role = 'client'
    ORDER BY id
");
$stmt->execute();
$users = $stmt->fetchAll();

$reportResults = [];

foreach ($users as $user) {
    try {
        require_once __DIR__ . '/reporting_enhanced.php';
        $reporting = new ReportingSystem();
        
        // Generate report for yesterday
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        $reportData = $reporting->generateMarketingReport($user['id'], $yesterday, $yesterday, 'json');
        
        if (!isset($reportData['error'])) {
            // Store daily report summary
            $stmt = $db->prepare("
                INSERT INTO daily_reports 
                (user_id, report_date, total_ad_spend, total_conversions, total_clicks, total_impressions, report_data)
                VALUES (?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                total_ad_spend = VALUES(total_ad_spend),
                total_conversions = VALUES(total_conversions),
                total_clicks = VALUES(total_clicks),
                total_impressions = VALUES(total_impressions),
                report_data = VALUES(report_data),
                updated_at = NOW()
            ");
            
            $stmt->execute([
                $user['id'],
                $yesterday,
                $reportData['summary']['ad_spend'],
                $reportData['summary']['total_conversions'],
                $reportData['summary']['total_clicks'],
                $reportData['summary']['total_impressions'],
                json_encode($reportData)
            ]);
            
            $reportResults[] = [
                'user_id' => $user['id'],
                'success' => true,
                'message' => 'Daily report generated successfully'
            ];
        } else {
            $reportResults[] = [
                'user_id' => $user['id'],
                'success' => false,
                'message' => $reportData['error']
            ];
        }
        
    } catch (Exception $e) {
        error_log("Daily report error for user {$user['id']}: " . $e->getMessage());
        $reportResults[] = [
            'user_id' => $user['id'],
            'success' => false,
            'message' => $e->getMessage()
        ];
    }
}

// Log report results
$logMessage = "Daily report generation completed at " . date('Y-m-d H:i:s') . "\n";
$logMessage .= "Users processed: " . count($users) . "\n";
$logMessage .= "Results: " . json_encode($reportResults) . "\n";

file_put_contents(__DIR__ . '/../logs/daily_reports.log', $logMessage, FILE_APPEND);

echo "Daily report generation completed. Check logs for details.\n";
?>
