<?php
/**
 * File Upload API
 * Handles file uploads for direct mail designs, mailing lists, etc.
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

// Create uploads directory if it doesn't exist
$uploadDir = __DIR__ . '/../uploads/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Create user-specific directory
$userUploadDir = $uploadDir . 'user_' . $userId . '/';
if (!is_dir($userUploadDir)) {
    mkdir($userUploadDir, 0755, true);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requireCSRFToken();
    
    $uploadType = $_POST['upload_type'] ?? 'other';
    $relatedId = isset($_POST['related_id']) ? intval($_POST['related_id']) : null;
    $relatedType = $_POST['related_type'] ?? null;
    
    // Validate upload type
    $allowedTypes = ['direct_mail_design', 'email_template', 'mailing_list', 'other'];
    if (!in_array($uploadType, $allowedTypes)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid upload type']);
        exit;
    }
    
    if (!isset($_FILES['file'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'No file uploaded']);
        exit;
    }
    
    $file = $_FILES['file'];
    
    // Validate file
    if ($file['error'] !== UPLOAD_ERR_OK) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'File upload error']);
        exit;
    }
    
    // Validate file size (max 10MB)
    $maxSize = 10 * 1024 * 1024; // 10MB
    if ($file['size'] > $maxSize) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'File size exceeds 10MB limit']);
        exit;
    }
    
    // Validate file type based on upload type
    $allowedExtensions = [];
    $allowedMimeTypes = [];
    
    switch ($uploadType) {
        case 'direct_mail_design':
            $allowedExtensions = ['jpg', 'jpeg', 'png', 'pdf'];
            $allowedMimeTypes = ['image/jpeg', 'image/png', 'application/pdf'];
            break;
        case 'email_template':
            $allowedExtensions = ['html', 'txt'];
            $allowedMimeTypes = ['text/html', 'text/plain'];
            break;
        case 'mailing_list':
            $allowedExtensions = ['csv', 'xlsx', 'xls'];
            $allowedMimeTypes = ['text/csv', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'];
            break;
        default:
            $allowedExtensions = ['jpg', 'jpeg', 'png', 'pdf', 'txt', 'csv', 'html'];
    }
    
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!empty($allowedExtensions) && !in_array($extension, $allowedExtensions)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid file type. Allowed: ' . implode(', ', $allowedExtensions)]);
        exit;
    }
    
    // Generate unique filename
    $fileName = uniqid() . '_' . time() . '.' . $extension;
    $filePath = $userUploadDir . $fileName;
    
    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $filePath)) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Failed to save file']);
        exit;
    }
    
    // Save file record to database
    $relativePath = 'uploads/user_' . $userId . '/' . $fileName;
    
    $stmt = $db->prepare("
        INSERT INTO file_uploads 
        (user_id, file_name, original_name, file_path, file_type, file_size, mime_type, upload_type, related_id, related_type)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $userId,
        $fileName,
        $file['name'],
        $relativePath,
        $extension,
        $file['size'],
        $file['type'],
        $uploadType,
        $relatedId,
        $relatedType
    ]);
    
    $uploadId = $db->lastInsertId();
    
    // Return success response
    header('Content-Type: application/json');
    echo json_encode([
        'success' => true,
        'message' => 'File uploaded successfully',
        'upload_id' => $uploadId,
        'file_path' => $relativePath,
        'file_name' => $file['name'],
        'file_size' => $file['size']
    ]);
    
} else {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
}

