<?php
/**
 * Enhanced Facebook Marketing API Integration
 * Requires: Facebook SDK for PHP
 * Installation: composer require facebook/php-business-sdk
 */

require_once __DIR__ . '/../config/facebook_config.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/credential_helper.php';

class FacebookAdsAPI {
    private $appId;
    private $appSecret;
    private $accessToken;
    private $apiVersion;
    private $adAccountId;
    
    public function __construct($userId = null) {
        $this->appId = FACEBOOK_APP_ID;
        $this->appSecret = FACEBOOK_APP_SECRET;
        $this->accessToken = FACEBOOK_ACCESS_TOKEN;
        $this->apiVersion = FACEBOOK_API_VERSION;
        
        // If user-specific credentials are provided, use those
        if ($userId) {
            $userCreds = getUserCredentials($userId, 'facebook_ads');
            if ($userCreds) {
                $this->appId = $userCreds['app_id'] ?? $this->appId;
                $this->appSecret = $userCreds['app_secret'] ?? $this->appSecret;
                $this->accessToken = $userCreds['access_token'] ?? $this->accessToken;
                $this->adAccountId = $userCreds['ad_account_id'] ?? null;
            }
        }
    }
    
    /**
     * Get ad performance data
     */
    public function getAdPerformance($adAccountId, $startDate, $endDate) {
        try {
            // Check if Facebook SDK is available
            if (!class_exists('FacebookAds\Api')) {
                return $this->getMockAdData($startDate, $endDate);
            }
            
            // Initialize Facebook API
            $api = $this->initializeFacebookAPI();
            
            // Get ad account
            $adAccount = new \FacebookAds\Object\AdAccount($adAccountId);
            
            // Get insights for the date range
            $insights = $adAccount->getInsights([
                'fields' => [
                    'campaign_id',
                    'campaign_name',
                    'adset_id',
                    'adset_name',
                    'ad_id',
                    'ad_name',
                    'impressions',
                    'clicks',
                    'ctr',
                    'cpc',
                    'spend',
                    'conversions',
                    'cost_per_conversion',
                    'conversion_rate',
                    'reach',
                    'frequency'
                ],
                'time_range' => [
                    'since' => $startDate,
                    'until' => $endDate
                ],
                'level' => 'campaign'
            ]);
            
            $campaigns = [];
            $summary = [
                'total_impressions' => 0,
                'total_clicks' => 0,
                'total_spend' => 0,
                'total_conversions' => 0,
                'average_ctr' => 0,
                'average_cpc' => 0,
                'average_conversion_rate' => 0,
                'total_reach' => 0,
                'average_frequency' => 0
            ];
            
            foreach ($insights as $insight) {
                $campaignData = [
                    'id' => $insight->campaign_id,
                    'name' => $insight->campaign_name,
                    'impressions' => $insight->impressions ?? 0,
                    'clicks' => $insight->clicks ?? 0,
                    'ctr' => $insight->ctr ?? 0,
                    'cpc' => $insight->cpc ?? 0,
                    'spend' => $insight->spend ?? 0,
                    'conversions' => $insight->conversions ?? 0,
                    'cost_per_conversion' => $insight->cost_per_conversion ?? 0,
                    'conversion_rate' => $insight->conversion_rate ?? 0,
                    'reach' => $insight->reach ?? 0,
                    'frequency' => $insight->frequency ?? 0
                ];
                
                $campaigns[] = $campaignData;
                
                // Update summary
                $summary['total_impressions'] += $campaignData['impressions'];
                $summary['total_clicks'] += $campaignData['clicks'];
                $summary['total_spend'] += $campaignData['spend'];
                $summary['total_conversions'] += $campaignData['conversions'];
                $summary['total_reach'] += $campaignData['reach'];
            }
            
            // Calculate averages
            if ($summary['total_clicks'] > 0) {
                $summary['average_ctr'] = ($summary['total_clicks'] / $summary['total_impressions']) * 100;
                $summary['average_cpc'] = $summary['total_spend'] / $summary['total_clicks'];
                $summary['average_conversion_rate'] = ($summary['total_conversions'] / $summary['total_clicks']) * 100;
            }
            if (count($campaigns) > 0) {
                $summary['average_frequency'] = array_sum(array_column($campaigns, 'frequency')) / count($campaigns);
            }
            
            return [
                'campaigns' => $campaigns,
                'summary' => $summary
            ];
            
        } catch (Exception $e) {
            error_log("Facebook Ads API Error: " . $e->getMessage());
            return $this->getMockAdData($startDate, $endDate);
        }
    }
    
    /**
     * Create a new Facebook ad campaign
     */
    public function createCampaign($adAccountId, $campaignData) {
        try {
            if (!class_exists('FacebookAds\Api')) {
                return ['error' => 'Facebook SDK not installed'];
            }
            
            $api = $this->initializeFacebookAPI();
            $adAccount = new \FacebookAds\Object\AdAccount($adAccountId);
            
            // Create campaign
            $campaign = new \FacebookAds\Object\Campaign();
            $campaign->setData([
                'name' => $campaignData['name'],
                'objective' => $campaignData['objective'] ?? 'CONVERSIONS',
                'status' => $campaignData['status'] ?? 'ACTIVE',
                'special_ad_categories' => $campaignData['special_ad_categories'] ?? []
            ]);
            
            $campaign->create([
                \FacebookAds\Object\Campaign::PARENT_ID => $adAccountId
            ]);
            
            // Create ad set
            $adSet = new \FacebookAds\Object\AdSet();
            $adSet->setData([
                'name' => $campaignData['name'] . ' Ad Set',
                'campaign_id' => $campaign->id,
                'daily_budget' => $campaignData['budget'] * 100, // Convert to cents
                'billing_event' => 'IMPRESSIONS',
                'optimization_goal' => 'CONVERSIONS',
                'bid_amount' => $campaignData['bid_amount'] ?? 100, // Default bid in cents
                'status' => 'ACTIVE',
                'targeting' => $campaignData['targeting'] ?? []
            ]);
            
            $adSet->create([
                \FacebookAds\Object\AdSet::PARENT_ID => $adAccountId
            ]);
            
            return [
                'success' => true,
                'campaign_id' => $campaign->id,
                'adset_id' => $adSet->id,
                'message' => 'Campaign created successfully'
            ];
            
        } catch (Exception $e) {
            error_log("Facebook Ads Campaign Creation Error: " . $e->getMessage());
            return ['error' => 'Failed to create campaign: ' . $e->getMessage()];
        }
    }
    
    /**
     * Update campaign budget
     */
    public function updateCampaignBudget($adAccountId, $campaignId, $newBudget) {
        try {
            if (!class_exists('FacebookAds\Api')) {
                return ['error' => 'Facebook SDK not installed'];
            }
            
            $api = $this->initializeFacebookAPI();
            
            // Get ad sets for the campaign
            $adSets = new \FacebookAds\Object\AdSet();
            $adSets = $adSets->getAdSets([
                'fields' => ['id', 'daily_budget'],
                'filtering' => [
                    ['field' => 'campaign.id', 'operator' => 'EQUAL', 'value' => $campaignId]
                ]
            ]);
            
            $updatedAdSets = [];
            foreach ($adSets as $adSet) {
                $adSet->update([
                    'daily_budget' => $newBudget * 100 // Convert to cents
                ]);
                $updatedAdSets[] = $adSet->id;
            }
            
            return [
                'success' => true,
                'message' => 'Budget updated successfully',
                'new_budget' => $newBudget,
                'updated_adsets' => $updatedAdSets
            ];
            
        } catch (Exception $e) {
            error_log("Facebook Ads Budget Update Error: " . $e->getMessage());
            return ['error' => 'Failed to update budget: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get audience insights and targeting suggestions
     */
    public function getAudienceInsights($adAccountId, $targeting = []) {
        try {
            if (!class_exists('FacebookAds\Api')) {
                return $this->getMockAudienceData();
            }
            
            $api = $this->initializeFacebookAPI();
            $adAccount = new \FacebookAds\Object\AdAccount($adAccountId);
            
            // Get reach estimate
            $reachEstimate = $adAccount->getReachEstimate([
                'targeting_spec' => [
                    'geo_locations' => $targeting['geo_locations'] ?? [],
                    'age_min' => $targeting['age_min'] ?? 18,
                    'age_max' => $targeting['age_max'] ?? 65,
                    'genders' => $targeting['genders'] ?? [1, 2], // 1 = male, 2 = female
                    'interests' => $targeting['interests'] ?? []
                ]
            ]);
            
            return [
                'estimated_reach' => $reachEstimate->users ?? 0,
                'targeting' => $targeting,
                'suggestions' => $this->getTargetingSuggestions($targeting)
            ];
            
        } catch (Exception $e) {
            error_log("Facebook Ads Audience Insights Error: " . $e->getMessage());
            return $this->getMockAudienceData();
        }
    }
    
    /**
     * Sync ad performance data to database
     */
    public function syncAdPerformance($userId) {
        try {
            $db = getDB();
            $adAccountId = $this->adAccountId;
            
            if (!$adAccountId) {
                return ['error' => 'Ad Account ID not configured'];
            }
            
            // Get performance data for last 30 days
            $endDate = date('Y-m-d');
            $startDate = date('Y-m-d', strtotime('-30 days'));
            
            $performanceData = $this->getAdPerformance($adAccountId, $startDate, $endDate);
            
            if (isset($performanceData['error'])) {
                return $performanceData;
            }
            
            // Store performance data in database
            foreach ($performanceData['campaigns'] as $campaign) {
                $stmt = $db->prepare("
                    INSERT INTO ad_performance 
                    (user_id, platform, campaign_id, campaign_name, date, impressions, clicks, cost, conversions, ctr, cpc, cost_per_conversion)
                    VALUES (?, 'facebook_ads', ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE
                    impressions = VALUES(impressions),
                    clicks = VALUES(clicks),
                    cost = VALUES(cost),
                    conversions = VALUES(conversions),
                    ctr = VALUES(ctr),
                    cpc = VALUES(cpc),
                    cost_per_conversion = VALUES(cost_per_conversion),
                    updated_at = NOW()
                ");
                
                $stmt->execute([
                    $userId,
                    $campaign['id'],
                    $campaign['name'],
                    $endDate,
                    $campaign['impressions'],
                    $campaign['clicks'],
                    $campaign['spend'],
                    $campaign['conversions'],
                    $campaign['ctr'],
                    $campaign['cpc'],
                    $campaign['cost_per_conversion']
                ]);
            }
            
            return [
                'success' => true,
                'message' => 'Performance data synced successfully',
                'campaigns_synced' => count($performanceData['campaigns'])
            ];
            
        } catch (Exception $e) {
            error_log("Facebook Ads Sync Error: " . $e->getMessage());
            return ['error' => 'Failed to sync performance data: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get budget alerts and monitoring
     */
    public function getBudgetAlerts($adAccountId) {
        try {
            if (!class_exists('FacebookAds\Api')) {
                return $this->getMockBudgetAlerts();
            }
            
            $api = $this->initializeFacebookAPI();
            $adAccount = new \FacebookAds\Object\AdAccount($adAccountId);
            
            // Get current day spend
            $insights = $adAccount->getInsights([
                'fields' => [
                    'campaign_id',
                    'campaign_name',
                    'spend',
                    'daily_budget'
                ],
                'time_range' => [
                    'since' => date('Y-m-d'),
                    'until' => date('Y-m-d')
                ],
                'level' => 'campaign'
            ]);
            
            $alerts = [];
            
            foreach ($insights as $insight) {
                $spentToday = $insight->spend ?? 0;
                $dailyBudget = $insight->daily_budget ?? 0;
                $usagePercentage = $dailyBudget > 0 ? ($spentToday / $dailyBudget) * 100 : 0;
                
                $alertType = null;
                if ($usagePercentage >= 90) {
                    $alertType = 'budget_critical';
                } elseif ($usagePercentage >= 75) {
                    $alertType = 'budget_warning';
                } elseif ($usagePercentage >= 50) {
                    $alertType = 'budget_info';
                }
                
                if ($alertType) {
                    $alerts[] = [
                        'campaign_id' => $insight->campaign_id,
                        'campaign_name' => $insight->campaign_name,
                        'budget_used' => round($usagePercentage, 2),
                        'daily_budget' => $dailyBudget,
                        'spent_today' => $spentToday,
                        'alert_type' => $alertType,
                        'date' => date('Y-m-d')
                    ];
                }
            }
            
            return ['alerts' => $alerts];
            
        } catch (Exception $e) {
            error_log("Facebook Ads Budget Alerts Error: " . $e->getMessage());
            return $this->getMockBudgetAlerts();
        }
    }
    
    /**
     * Initialize Facebook API
     */
    private function initializeFacebookAPI() {
        \FacebookAds\Api::init($this->appId, $this->appSecret, $this->accessToken);
        return \FacebookAds\Api::instance();
    }
    
    /**
     * Get targeting suggestions
     */
    private function getTargetingSuggestions($targeting) {
        // This would typically use Facebook's targeting suggestions API
        return [
            'interests' => [
                'Restaurants',
                'Food & Dining',
                'Local Business',
                'Indian Cuisine'
            ],
            'behaviors' => [
                'Frequent travelers',
                'Food enthusiasts',
                'Local business supporters'
            ],
            'demographics' => [
                'Age: 25-54',
                'Income: $40k+',
                'Education: College+'
            ]
        ];
    }
    
    /**
     * Get mock ad data for testing
     */
    private function getMockAdData($startDate, $endDate) {
        return [
            'campaigns' => [
                [
                    'id' => '120123456789',
                    'name' => 'Sample Facebook Campaign',
                    'impressions' => 25000,
                    'clicks' => 800,
                    'ctr' => 3.2,
                    'cpc' => 0.75,
                    'spend' => 600.00,
                    'conversions' => 15,
                    'cost_per_conversion' => 40.00,
                    'conversion_rate' => 1.875,
                    'reach' => 20000,
                    'frequency' => 1.25
                ],
                [
                    'id' => '120987654321',
                    'name' => 'Sample Instagram Campaign',
                    'impressions' => 18000,
                    'clicks' => 450,
                    'ctr' => 2.5,
                    'cpc' => 0.90,
                    'spend' => 405.00,
                    'conversions' => 9,
                    'cost_per_conversion' => 45.00,
                    'conversion_rate' => 2.0,
                    'reach' => 15000,
                    'frequency' => 1.2
                ]
            ],
            'summary' => [
                'total_impressions' => 43000,
                'total_clicks' => 1250,
                'total_spend' => 1005.00,
                'total_conversions' => 24,
                'average_ctr' => 2.91,
                'average_cpc' => 0.804,
                'average_conversion_rate' => 1.92,
                'total_reach' => 35000,
                'average_frequency' => 1.23
            ]
        ];
    }
    
    /**
     * Get mock audience data for testing
     */
    private function getMockAudienceData() {
        return [
            'estimated_reach' => 50000,
            'targeting' => [
                'age_min' => 25,
                'age_max' => 54,
                'genders' => [1, 2],
                'interests' => ['Restaurants', 'Food & Dining']
            ],
            'suggestions' => [
                'interests' => [
                    'Restaurants',
                    'Food & Dining',
                    'Local Business',
                    'Indian Cuisine'
                ],
                'behaviors' => [
                    'Frequent travelers',
                    'Food enthusiasts',
                    'Local business supporters'
                ],
                'demographics' => [
                    'Age: 25-54',
                    'Income: $40k+',
                    'Education: College+'
                ]
            ]
        ];
    }
    
    /**
     * Get mock budget alerts for testing
     */
    private function getMockBudgetAlerts() {
        return [
            'alerts' => [
                [
                    'campaign_id' => '120123456789',
                    'campaign_name' => 'Sample Facebook Campaign',
                    'budget_used' => 78.5,
                    'daily_budget' => 50.00,
                    'spent_today' => 39.25,
                    'alert_type' => 'budget_warning',
                    'date' => date('Y-m-d')
                ],
                [
                    'campaign_id' => '120987654321',
                    'campaign_name' => 'Sample Instagram Campaign',
                    'budget_used' => 92.0,
                    'daily_budget' => 30.00,
                    'spent_today' => 27.60,
                    'alert_type' => 'budget_critical',
                    'date' => date('Y-m-d')
                ]
            ]
        ];
    }
}

// API endpoint for Facebook Ads operations
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    require_once __DIR__ . '/../includes/auth.php';
    requireClient();
    
    $userId = $_SESSION['user_id'];
    $facebookAds = new FacebookAdsAPI($userId);
    $action = $_POST['action'];
    
    header('Content-Type: application/json');
    
    switch ($action) {
        case 'sync_performance':
            $result = $facebookAds->syncAdPerformance($userId);
            echo json_encode($result);
            break;
            
        case 'get_budget_alerts':
            $adAccountId = $_POST['ad_account_id'] ?? null;
            if (!$adAccountId) {
                echo json_encode(['error' => 'Ad Account ID required']);
                break;
            }
            $result = $facebookAds->getBudgetAlerts($adAccountId);
            echo json_encode($result);
            break;
            
        case 'create_campaign':
            $adAccountId = $_POST['ad_account_id'] ?? null;
            $campaignData = [
                'name' => $_POST['campaign_name'] ?? '',
                'budget' => floatval($_POST['budget'] ?? 0),
                'objective' => $_POST['objective'] ?? 'CONVERSIONS',
                'status' => $_POST['status'] ?? 'ACTIVE',
                'targeting' => json_decode($_POST['targeting'] ?? '{}', true)
            ];
            
            if (!$adAccountId || empty($campaignData['name']) || $campaignData['budget'] <= 0) {
                echo json_encode(['error' => 'Invalid campaign data']);
                break;
            }
            
            $result = $facebookAds->createCampaign($adAccountId, $campaignData);
            echo json_encode($result);
            break;
            
        case 'update_budget':
            $adAccountId = $_POST['ad_account_id'] ?? null;
            $campaignId = $_POST['campaign_id'] ?? null;
            $newBudget = floatval($_POST['new_budget'] ?? 0);
            
            if (!$adAccountId || !$campaignId || $newBudget <= 0) {
                echo json_encode(['error' => 'Invalid budget update data']);
                break;
            }
            
            $result = $facebookAds->updateCampaignBudget($adAccountId, $campaignId, $newBudget);
            echo json_encode($result);
            break;
            
        case 'get_audience_insights':
            $adAccountId = $_POST['ad_account_id'] ?? null;
            $targeting = json_decode($_POST['targeting'] ?? '{}', true);
            
            if (!$adAccountId) {
                echo json_encode(['error' => 'Ad Account ID required']);
                break;
            }
            
            $result = $facebookAds->getAudienceInsights($adAccountId, $targeting);
            echo json_encode($result);
            break;
            
        default:
            echo json_encode(['error' => 'Invalid action']);
    }
    exit;
}
?>
