<?php
/**
 * Email Template Builder API
 * Handles email template creation, editing, and management
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');
requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if ($action === 'list') {
                $includePublic = isset($_GET['include_public']) ? (bool)$_GET['include_public'] : true;
                $category = $_GET['category'] ?? null;
                
                $query = "SELECT * FROM email_templates WHERE (user_id = ? OR (is_public = 1 AND ?))";
                $params = [$userId, $includePublic ? 1 : 0];
                
                if ($category) {
                    $query .= " AND template_category = ?";
                    $params[] = $category;
                }
                
                $query .= " ORDER BY is_public DESC, created_at DESC";
                
                $stmt = $db->prepare($query);
                $stmt->execute($params);
                $templates = $stmt->fetchAll();
                
                echo json_encode(['success' => true, 'data' => $templates]);
            } elseif ($action === 'get') {
                $templateId = intval($_GET['id'] ?? 0);
                
                $stmt = $db->prepare("SELECT * FROM email_templates WHERE id = ? AND (user_id = ? OR is_public = 1)");
                $stmt->execute([$templateId, $userId]);
                $template = $stmt->fetch();
                
                if (!$template) {
                    throw new Exception('Template not found');
                }
                
                echo json_encode(['success' => true, 'data' => $template]);
            }
            break;
            
        case 'POST':
            requireCSRFToken();
            
            if ($action === 'create' || $action === 'update') {
                $templateId = $action === 'update' ? intval($_POST['id'] ?? 0) : null;
                $templateName = trim($_POST['template_name'] ?? '');
                $subjectLine = $_POST['subject_line'] ?? '';
                $htmlContent = $_POST['html_content'] ?? '';
                $textContent = $_POST['text_content'] ?? '';
                $category = $_POST['category'] ?? null;
                $isPublic = isset($_POST['is_public']) ? (bool)$_POST['is_public'] : false;
                
                if (empty($templateName) || empty($htmlContent)) {
                    throw new Exception('Template name and HTML content are required');
                }
                
                // Only admins can create public templates
                if ($isPublic && !hasRole('admin')) {
                    $isPublic = false;
                }
                
                if ($templateId) {
                    // Update existing
                    $stmt = $db->prepare("SELECT user_id FROM email_templates WHERE id = ?");
                    $stmt->execute([$templateId]);
                    $existing = $stmt->fetch();
                    
                    if (!$existing || ($existing['user_id'] != $userId && !hasRole('admin'))) {
                        throw new Exception('Template not found or access denied');
                    }
                    
                    $stmt = $db->prepare("
                        UPDATE email_templates 
                        SET template_name = ?, subject_line = ?, html_content = ?, 
                            text_content = ?, template_category = ?, is_public = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$templateName, $subjectLine, $htmlContent, $textContent, $category, $isPublic ? 1 : 0, $templateId]);
                } else {
                    // Create new
                    $stmt = $db->prepare("
                        INSERT INTO email_templates 
                        (user_id, template_name, subject_line, html_content, text_content, template_category, is_public)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$userId, $templateName, $subjectLine, $htmlContent, $textContent, $category, $isPublic ? 1 : 0]);
                    $templateId = $db->lastInsertId();
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => $action === 'update' ? 'Template updated' : 'Template created',
                    'template_id' => $templateId
                ]);
                
            } elseif ($action === 'use') {
                // Use template for email campaign
                $templateId = intval($_POST['template_id'] ?? 0);
                
                $stmt = $db->prepare("SELECT * FROM email_templates WHERE id = ? AND (user_id = ? OR is_public = 1)");
                $stmt->execute([$templateId, $userId]);
                $template = $stmt->fetch();
                
                if (!$template) {
                    throw new Exception('Template not found');
                }
                
                // Update usage count
                $stmt = $db->prepare("UPDATE email_templates SET usage_count = usage_count + 1 WHERE id = ?");
                $stmt->execute([$templateId]);
                
                echo json_encode(['success' => true, 'data' => $template]);
            }
            break;
            
        case 'DELETE':
            requireCSRFToken();
            
            if ($action === 'delete') {
                $templateId = intval($_GET['id'] ?? $_POST['id'] ?? 0);
                
                $stmt = $db->prepare("SELECT user_id FROM email_templates WHERE id = ?");
                $stmt->execute([$templateId]);
                $template = $stmt->fetch();
                
                if (!$template || ($template['user_id'] != $userId && !hasRole('admin'))) {
                    throw new Exception('Template not found or access denied');
                }
                
                $stmt = $db->prepare("DELETE FROM email_templates WHERE id = ?");
                $stmt->execute([$templateId]);
                
                echo json_encode(['success' => true, 'message' => 'Template deleted']);
            }
            break;
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

