<?php
/**
 * Enhanced Email Service Integration (SendGrid/Mailchimp)
 * Requires: SendGrid PHP SDK and Mailchimp API
 * Installation: composer require sendgrid/sendgrid mailchimp/marketing
 */

require_once __DIR__ . '/../config/email_config.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/credential_helper.php';

class EmailServiceEnhanced {
    private $service;
    private $apiKey;
    private $fromEmail;
    private $fromName;
    private $serverPrefix;
    
    public function __construct($service = 'sendgrid', $userId = null) {
        $this->service = $service;
        
        if ($service === 'sendgrid') {
            $this->apiKey = SENDGRID_API_KEY;
            $this->fromEmail = SENDGRID_FROM_EMAIL;
            $this->fromName = SENDGRID_FROM_NAME;
        } elseif ($service === 'mailchimp') {
            $this->apiKey = MAILCHIMP_API_KEY;
            $this->serverPrefix = MAILCHIMP_SERVER_PREFIX;
        }
        
        // Get user-specific credentials if provided
        if ($userId) {
            $serviceName = $this->service === 'sendgrid' ? 'sendgrid' : 'mailchimp';
            $userCreds = getUserCredentials($userId, $serviceName);
            if ($userCreds) {
                $this->apiKey = $userCreds['api_key'] ?? $this->apiKey;
                if ($serviceName === 'sendgrid') {
                    $this->fromEmail = $userCreds['from_email'] ?? SENDGRID_FROM_EMAIL;
                    $this->fromName = $userCreds['from_name'] ?? SENDGRID_FROM_NAME;
                } elseif ($serviceName === 'mailchimp') {
                    $this->serverPrefix = $userCreds['server_prefix'] ?? '';
                }
            }
        }
    }
    
    /**
     * Send email campaign
     */
    public function sendEmailCampaign($campaignData) {
        try {
            if ($this->service === 'sendgrid') {
                return $this->sendWithSendGrid($campaignData);
            } elseif ($this->service === 'mailchimp') {
                return $this->sendWithMailchimp($campaignData);
            }
            
            return ['error' => 'Invalid email service'];
            
        } catch (Exception $e) {
            error_log("Email Service Error: " . $e->getMessage());
            return ['error' => 'Failed to send email: ' . $e->getMessage()];
        }
    }
    
    /**
     * Send email using SendGrid
     */
    private function sendWithSendGrid($campaignData) {
        if (!class_exists('SendGrid')) {
            return ['error' => 'SendGrid SDK not installed'];
        }
        
        $email = new \SendGrid\Mail\Mail();
        $email->setFrom($this->fromEmail, $this->fromName);
        $email->setSubject($campaignData['subject']);
        
        // Add recipients
        if (isset($campaignData['to'])) {
            if (is_array($campaignData['to'])) {
                foreach ($campaignData['to'] as $recipient) {
                    $email->addTo($recipient['email'], $recipient['name'] ?? '');
                }
            } else {
                $email->addTo($campaignData['to']);
            }
        }
        
        // Set content
        if (isset($campaignData['html_content'])) {
            $email->addContent("text/html", $campaignData['html_content']);
        }
        if (isset($campaignData['text_content'])) {
            $email->addContent("text/plain", $campaignData['text_content']);
        }
        
        // Add tracking
        $email->setTrackingSettings([
            'click_tracking' => [
                'enable' => true,
                'enable_text' => true
            ],
            'open_tracking' => [
                'enable' => true
            ]
        ]);
        
        // Add custom args for tracking
        if (isset($campaignData['campaign_id'])) {
            $email->addCustomArg('campaign_id', $campaignData['campaign_id']);
        }
        if (isset($campaignData['user_id'])) {
            $email->addCustomArg('user_id', $campaignData['user_id']);
        }
        
        $sendgrid = new \SendGrid($this->apiKey);
        $response = $sendgrid->send($email);
        
        if ($response->statusCode() >= 200 && $response->statusCode() < 300) {
            return [
                'success' => true,
                'message_id' => $response->headers()['X-Message-Id'] ?? null,
                'message' => 'Email sent successfully'
            ];
        } else {
            return [
                'error' => 'SendGrid API error: ' . $response->statusCode(),
                'body' => $response->body()
            ];
        }
    }
    
    /**
     * Send email using Mailchimp
     */
    private function sendWithMailchimp($campaignData) {
        if (!class_exists('MailchimpMarketing\ApiClient')) {
            return ['error' => 'Mailchimp SDK not installed'];
        }
        
        $client = new \MailchimpMarketing\ApiClient();
        $client->setConfig([
            'apiKey' => $this->apiKey,
            'server' => $this->serverPrefix
        ]);
        
        // Create campaign
        $campaign = $client->campaigns->create([
            'type' => 'regular',
            'recipients' => [
                'list_id' => $campaignData['list_id'] ?? '',
                'segment_opts' => $campaignData['segment_opts'] ?? []
            ],
            'settings' => [
                'subject_line' => $campaignData['subject'],
                'from_name' => $this->fromName ?? 'Marketing Team',
                'reply_to' => $this->fromEmail ?? 'noreply@example.com',
                'title' => $campaignData['title'] ?? $campaignData['subject']
            ]
        ]);
        
        // Set content
        $client->campaigns->setContent($campaign['id'], [
            'html' => $campaignData['html_content'] ?? '',
            'plain_text' => $campaignData['text_content'] ?? ''
        ]);
        
        // Send campaign
        $sendResult = $client->campaigns->send($campaign['id']);
        
        return [
            'success' => true,
            'campaign_id' => $campaign['id'],
            'message' => 'Email campaign sent successfully'
        ];
    }
    
    /**
     * Create email template
     */
    public function createTemplate($templateData) {
        try {
            if ($this->service === 'sendgrid') {
                return $this->createSendGridTemplate($templateData);
            } elseif ($this->service === 'mailchimp') {
                return $this->createMailchimpTemplate($templateData);
            }
            
            return ['error' => 'Invalid email service'];
            
        } catch (Exception $e) {
            error_log("Template Creation Error: " . $e->getMessage());
            return ['error' => 'Failed to create template: ' . $e->getMessage()];
        }
    }
    
    /**
     * Create SendGrid template
     */
    private function createSendGridTemplate($templateData) {
        if (!class_exists('SendGrid')) {
            return ['error' => 'SendGrid SDK not installed'];
        }
        
        $sendgrid = new \SendGrid($this->apiKey);
        
        $template = new \SendGrid\Mail\Template();
        $template->setName($templateData['name']);
        $template->setGeneration('dynamic');
        
        $response = $sendgrid->client->templates()->post($template);
        
        if ($response->statusCode() >= 200 && $response->statusCode() < 300) {
            $templateId = json_decode($response->body())->id;
            
            // Create template version
            $version = new \SendGrid\Mail\TemplateVersion();
            $version->setTemplateId($templateId);
            $version->setActive(1);
            $version->setName($templateData['name'] . ' Version 1');
            $version->setHtmlContent($templateData['html_content']);
            $version->setPlainContent($templateData['text_content']);
            
            $versionResponse = $sendgrid->client->templates()->versions()->post($version);
            
            if ($versionResponse->statusCode() >= 200 && $versionResponse->statusCode() < 300) {
                return [
                    'success' => true,
                    'template_id' => $templateId,
                    'message' => 'Template created successfully'
                ];
            }
        }
        
        return ['error' => 'Failed to create SendGrid template'];
    }
    
    /**
     * Create Mailchimp template
     */
    private function createMailchimpTemplate($templateData) {
        if (!class_exists('MailchimpMarketing\ApiClient')) {
            return ['error' => 'Mailchimp SDK not installed'];
        }
        
        $client = new \MailchimpMarketing\ApiClient();
        $client->setConfig([
            'apiKey' => $this->apiKey,
            'server' => $this->serverPrefix
        ]);
        
        $template = $client->templates->create([
            'name' => $templateData['name'],
            'html' => $templateData['html_content'],
            'folder_id' => $templateData['folder_id'] ?? null
        ]);
        
        return [
            'success' => true,
            'template_id' => $template['id'],
            'message' => 'Template created successfully'
        ];
    }
    
    /**
     * Get campaign statistics
     */
    public function getCampaignStats($campaignId) {
        try {
            if ($this->service === 'sendgrid') {
                return $this->getSendGridStats($campaignId);
            } elseif ($this->service === 'mailchimp') {
                return $this->getMailchimpStats($campaignId);
            }
            
            return ['error' => 'Invalid email service'];
            
        } catch (Exception $e) {
            error_log("Email Stats Error: " . $e->getMessage());
            return ['error' => 'Failed to get campaign stats: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get SendGrid campaign statistics
     */
    private function getSendGridStats($campaignId) {
        if (!class_exists('SendGrid')) {
            return ['error' => 'SendGrid SDK not installed'];
        }
        
        $sendgrid = new \SendGrid($this->apiKey);
        
        // Get stats for the campaign
        $queryParams = [
            'start_date' => date('Y-m-d', strtotime('-30 days')),
            'end_date' => date('Y-m-d'),
            'aggregated_by' => 'day'
        ];
        
        $response = $sendgrid->client->stats()->get(null, $queryParams);
        
        if ($response->statusCode() >= 200 && $response->statusCode() < 300) {
            $stats = json_decode($response->body(), true);
            
            $totalStats = [
                'delivered' => 0,
                'opens' => 0,
                'clicks' => 0,
                'bounces' => 0,
                'spam_reports' => 0,
                'unsubscribes' => 0
            ];
            
            foreach ($stats as $stat) {
                $totalStats['delivered'] += $stat['stats'][0]['metrics']['delivered'] ?? 0;
                $totalStats['opens'] += $stat['stats'][0]['metrics']['opens'] ?? 0;
                $totalStats['clicks'] += $stat['stats'][0]['metrics']['clicks'] ?? 0;
                $totalStats['bounces'] += $stat['stats'][0]['metrics']['bounces'] ?? 0;
                $totalStats['spam_reports'] += $stat['stats'][0]['metrics']['spam_reports'] ?? 0;
                $totalStats['unsubscribes'] += $stat['stats'][0]['metrics']['unsubscribes'] ?? 0;
            }
            
            // Calculate rates
            $delivered = $totalStats['delivered'];
            $openRate = $delivered > 0 ? ($totalStats['opens'] / $delivered) * 100 : 0;
            $clickRate = $delivered > 0 ? ($totalStats['clicks'] / $delivered) * 100 : 0;
            $bounceRate = $delivered > 0 ? ($totalStats['bounces'] / $delivered) * 100 : 0;
            
            return [
                'success' => true,
                'stats' => array_merge($totalStats, [
                    'open_rate' => round($openRate, 2),
                    'click_rate' => round($clickRate, 2),
                    'bounce_rate' => round($bounceRate, 2)
                ])
            ];
        }
        
        return ['error' => 'Failed to get SendGrid stats'];
    }
    
    /**
     * Get Mailchimp campaign statistics
     */
    private function getMailchimpStats($campaignId) {
        if (!class_exists('MailchimpMarketing\ApiClient')) {
            return ['error' => 'Mailchimp SDK not installed'];
        }
        
        $client = new \MailchimpMarketing\ApiClient();
        $client->setConfig([
            'apiKey' => $this->apiKey,
            'server' => $this->serverPrefix
        ]);
        
        $report = $client->reports->getCampaignReport($campaignId);
        
        $stats = [
            'delivered' => $report['emails_sent'] ?? 0,
            'opens' => $report['opens']['unique_opens'] ?? 0,
            'clicks' => $report['clicks']['unique_clicks'] ?? 0,
            'bounces' => $report['bounces']['hard_bounces'] + $report['bounces']['soft_bounces'] ?? 0,
            'unsubscribes' => $report['unsubscribed'] ?? 0,
            'open_rate' => $report['opens']['open_rate'] ?? 0,
            'click_rate' => $report['clicks']['click_rate'] ?? 0,
            'bounce_rate' => $report['bounces']['bounce_rate'] ?? 0
        ];
        
        return [
            'success' => true,
            'stats' => $stats
        ];
    }
    
    /**
     * Sync email campaign data to database
     */
    public function syncEmailCampaign($userId, $campaignData) {
        try {
            $db = getDB();
            
            // Store campaign data
            $stmt = $db->prepare("
                INSERT INTO email_campaigns 
                (user_id, campaign_name, subject, recipient_count, sent_at, status, service)
                VALUES (?, ?, ?, ?, NOW(), 'sent', ?)
                ON DUPLICATE KEY UPDATE
                recipient_count = VALUES(recipient_count),
                sent_at = VALUES(sent_at),
                status = VALUES(status),
                updated_at = NOW()
            ");
            
            $recipientCount = is_array($campaignData['to']) ? count($campaignData['to']) : 1;
            
            $stmt->execute([
                $userId,
                $campaignData['campaign_name'] ?? 'Email Campaign',
                $campaignData['subject'],
                $recipientCount,
                $this->service
            ]);
            
            $campaignId = $db->lastInsertId();
            
            // Get and store campaign stats
            $stats = $this->getCampaignStats($campaignId);
            if (isset($stats['success']) && $stats['success']) {
                $this->storeEmailStats($userId, $campaignId, $stats['stats']);
            }
            
            return [
                'success' => true,
                'campaign_id' => $campaignId,
                'message' => 'Email campaign synced successfully'
            ];
            
        } catch (Exception $e) {
            error_log("Email Campaign Sync Error: " . $e->getMessage());
            return ['error' => 'Failed to sync email campaign: ' . $e->getMessage()];
        }
    }
    
    /**
     * Store email campaign statistics
     */
    private function storeEmailStats($userId, $campaignId, $stats) {
        try {
            $db = getDB();
            
            $stmt = $db->prepare("
                INSERT INTO email_campaign_stats 
                (user_id, campaign_id, delivered, opens, clicks, bounces, unsubscribes, open_rate, click_rate, bounce_rate, date)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURDATE())
                ON DUPLICATE KEY UPDATE
                delivered = VALUES(delivered),
                opens = VALUES(opens),
                clicks = VALUES(clicks),
                bounces = VALUES(bounces),
                unsubscribes = VALUES(unsubscribes),
                open_rate = VALUES(open_rate),
                click_rate = VALUES(click_rate),
                bounce_rate = VALUES(bounce_rate),
                updated_at = NOW()
            ");
            
            $stmt->execute([
                $userId,
                $campaignId,
                $stats['delivered'] ?? 0,
                $stats['opens'] ?? 0,
                $stats['clicks'] ?? 0,
                $stats['bounces'] ?? 0,
                $stats['unsubscribes'] ?? 0,
                $stats['open_rate'] ?? 0,
                $stats['click_rate'] ?? 0,
                $stats['bounce_rate'] ?? 0
            ]);
            
        } catch (Exception $e) {
            error_log("Email Stats Storage Error: " . $e->getMessage());
        }
    }
    
    /**
     * Get mock email data for testing
     */
    private function getMockEmailData() {
        return [
            'campaigns' => [
                [
                    'id' => 'email_001',
                    'name' => 'Weekly Newsletter',
                    'subject' => 'This Week at Taste of India',
                    'delivered' => 1250,
                    'opens' => 312,
                    'clicks' => 78,
                    'bounces' => 12,
                    'unsubscribes' => 3,
                    'open_rate' => 24.96,
                    'click_rate' => 6.24,
                    'bounce_rate' => 0.96
                ],
                [
                    'id' => 'email_002',
                    'name' => 'Special Promotion',
                    'subject' => '20% Off Your Next Order!',
                    'delivered' => 850,
                    'opens' => 255,
                    'clicks' => 127,
                    'bounces' => 8,
                    'unsubscribes' => 2,
                    'open_rate' => 30.0,
                    'click_rate' => 14.94,
                    'bounce_rate' => 0.94
                ]
            ],
            'summary' => [
                'total_delivered' => 2100,
                'total_opens' => 567,
                'total_clicks' => 205,
                'total_bounces' => 20,
                'total_unsubscribes' => 5,
                'average_open_rate' => 27.0,
                'average_click_rate' => 9.76,
                'average_bounce_rate' => 0.95
            ]
        ];
    }
}

// API endpoint for Email Service operations
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    require_once __DIR__ . '/../includes/auth.php';
    requireClient();
    
    $userId = $_SESSION['user_id'];
    $service = $_POST['service'] ?? 'sendgrid';
    $emailService = new EmailServiceEnhanced($service, $userId);
    $action = $_POST['action'];
    
    header('Content-Type: application/json');
    
    switch ($action) {
        case 'send_campaign':
            $campaignData = [
                'campaign_name' => $_POST['campaign_name'] ?? '',
                'subject' => $_POST['subject'] ?? '',
                'html_content' => $_POST['html_content'] ?? '',
                'text_content' => $_POST['text_content'] ?? '',
                'to' => json_decode($_POST['recipients'] ?? '[]', true),
                'campaign_id' => $_POST['campaign_id'] ?? null,
                'user_id' => $userId
            ];
            
            if (empty($campaignData['subject']) || empty($campaignData['html_content'])) {
                echo json_encode(['error' => 'Subject and content are required']);
                break;
            }
            
            $result = $emailService->sendEmailCampaign($campaignData);
            echo json_encode($result);
            break;
            
        case 'create_template':
            $templateData = [
                'name' => $_POST['template_name'] ?? '',
                'html_content' => $_POST['html_content'] ?? '',
                'text_content' => $_POST['text_content'] ?? '',
                'folder_id' => $_POST['folder_id'] ?? null
            ];
            
            if (empty($templateData['name']) || empty($templateData['html_content'])) {
                echo json_encode(['error' => 'Template name and content are required']);
                break;
            }
            
            $result = $emailService->createTemplate($templateData);
            echo json_encode($result);
            break;
            
        case 'get_campaign_stats':
            $campaignId = $_POST['campaign_id'] ?? null;
            if (!$campaignId) {
                echo json_encode(['error' => 'Campaign ID required']);
                break;
            }
            
            $result = $emailService->getCampaignStats($campaignId);
            echo json_encode($result);
            break;
            
        case 'sync_campaign':
            $campaignData = [
                'campaign_name' => $_POST['campaign_name'] ?? '',
                'subject' => $_POST['subject'] ?? '',
                'to' => json_decode($_POST['recipients'] ?? '[]', true)
            ];
            
            $result = $emailService->syncEmailCampaign($userId, $campaignData);
            echo json_encode($result);
            break;
            
        default:
            echo json_encode(['error' => 'Invalid action']);
    }
    exit;
}
?>
