<?php
/**
 * Email Service Integration (SMTP/SendGrid/Mailchimp)
 */

// Start output buffering to prevent any accidental output from corrupting JSON responses
ob_start();

require_once __DIR__ . '/../config/email_config.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/credential_helper.php';
require_once __DIR__ . '/../includes/Mailer.php';
require_once __DIR__ . '/../includes/webhook_helper.php';

class EmailService {
    private $service;
    private $apiKey;
    
    public function __construct($userId = null) {
        $this->service = EMAIL_SERVICE;
        
        if ($this->service === 'smtp') {
            // SMTP will use Mailer with per-user overrides if available
            $this->apiKey = null;
        } elseif ($this->service === 'sendgrid') {
            $this->apiKey = SENDGRID_API_KEY;
        } elseif ($this->service === 'mailchimp') {
            $this->apiKey = MAILCHIMP_API_KEY;
        }
        
        // Get user-specific credentials if provided
        if ($userId) {
            $serviceName = $this->service === 'sendgrid' ? 'sendgrid' : ($this->service === 'mailchimp' ? 'mailchimp' : 'smtp');
            $userCreds = getUserCredentials($userId, $serviceName);
            if ($userCreds) {
                $this->apiKey = $userCreds['api_key'] ?? $this->apiKey;
                if ($serviceName === 'sendgrid') {
                    $this->fromEmail = $userCreds['from_email'] ?? SENDGRID_FROM_EMAIL;
                    $this->fromName = $userCreds['from_name'] ?? SENDGRID_FROM_NAME;
                } elseif ($serviceName === 'mailchimp') {
                    $this->serverPrefix = $userCreds['server_prefix'] ?? '';
                } elseif ($serviceName === 'smtp') {
                    $this->smtpOverrides = [
                        'host' => $userCreds['host'] ?? SMTP_HOST,
                        'port' => (int)($userCreds['port'] ?? SMTP_PORT),
                        'username' => $userCreds['username'] ?? SMTP_USER,
                        'password' => $userCreds['password'] ?? SMTP_PASS,
                        'secure' => $userCreds['secure'] ?? SMTP_SECURE,
                        'from_email' => $userCreds['from_email'] ?? SMTP_FROM_EMAIL,
                        'from_name' => $userCreds['from_name'] ?? SMTP_FROM_NAME,
                    ];
                }
            }
        }
    }
    
    /**
     * Send email campaign
     */
    public function sendCampaign($campaignId, $to = [], $subject, $content, $fromEmail = null, $fromName = null) {
        $db = getDB();
        
        if ($this->service === 'smtp') {
            return $this->sendViaSMTP($campaignId, $to, $subject, $content, $fromEmail, $fromName);
        } elseif ($this->service === 'sendgrid') {
            return $this->sendViaSendGrid($campaignId, $to, $subject, $content, $fromEmail, $fromName);
        } elseif ($this->service === 'mailchimp') {
            return $this->sendViaMailchimp($campaignId, $to, $subject, $content, $fromEmail, $fromName);
        }
        
        return false;
    }
    
    private function sendViaSMTP($campaignId, $to, $subject, $content, $fromEmail, $fromName) {
        $overrides = $this->smtpOverrides ?? [];
        if ($fromEmail) { $overrides['from_email'] = $fromEmail; }
        if ($fromName) { $overrides['from_name'] = $fromName; }

        $mailer = new Mailer($overrides);
        $result = $mailer->send([
            'to' => $to,
            'subject' => $subject,
            'html' => $content,
        ]);

        if (!empty($result['success'])) {
            $db = getDB();
            $stmt = $db->prepare("\n                UPDATE email_campaigns \n                SET sent_count = sent_count + ?, status = 'sent', sent_at = NOW()\n                WHERE id = ?\n            ");
            $recipientCount = is_array($to) ? count($to) : 1;
            $stmt->execute([$recipientCount, $campaignId]);
            
            // Trigger webhook for email sent
            if (isset($_SESSION['user_id'])) {
                triggerWebhookEvent('email.sent', [
                    'campaign_id' => $campaignId,
                    'recipient_count' => $recipientCount,
                    'subject' => $subject,
                    'sent_at' => date('c')
                ]);
            }
            
            return true;
        }

        error_log('SMTP send error: ' . ($result['error'] ?? 'unknown'));
        return false;
    }
    
    private function sendViaSendGrid($campaignId, $to, $subject, $content, $fromEmail, $fromName) {
        $fromEmail = $fromEmail ?: SENDGRID_FROM_EMAIL;
        $fromName = $fromName ?: SENDGRID_FROM_NAME;
        
        // Placeholder - requires SendGrid PHP library
        // Installation: composer require sendgrid/sendgrid
        
        $email = new \SendGrid\Mail\Mail();
        $email->setFrom($fromEmail, $fromName);
        $email->setSubject($subject);
        $email->addContent("text/html", $content);
        
        foreach ($to as $recipient) {
            $email->addTo($recipient);
        }
        
        $sendgrid = new \SendGrid($this->apiKey);
        
        try {
            $response = $sendgrid->send($email);
            
            // Update email campaign stats
            $db = getDB();
            $stmt = $db->prepare("
                UPDATE email_campaigns 
                SET sent_count = sent_count + ?, status = 'sent', sent_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([count($to), $campaignId]);
            
            // Trigger webhook for email sent
            if (isset($_SESSION['user_id'])) {
                triggerWebhookEvent('email.sent', [
                    'campaign_id' => $campaignId,
                    'recipient_count' => count($to),
                    'subject' => $subject,
                    'sent_at' => date('c')
                ]);
            }
            
            return true;
        } catch (Exception $e) {
            error_log("SendGrid error: " . $e->getMessage());
            return false;
        }
    }
    
    private function sendViaMailchimp($campaignId, $to, $subject, $content, $fromEmail, $fromName) {
        // Placeholder - requires Mailchimp PHP library
        return false;
    }
    
    /**
     * Track email opens/clicks via webhook
     */
    public function handleWebhook($data) {
        $db = getDB();
        
        // Process webhook data from SendGrid/Mailchimp
        // Update email_campaigns table with open/click stats
        
        // Trigger webhook events for opens/clicks
        if (isset($data['event']) && isset($data['email_campaign_id'])) {
            if ($data['event'] === 'open' || $data['event'] === 'click') {
                $eventType = $data['event'] === 'open' ? 'email.opened' : 'email.clicked';
                if (isset($_SESSION['user_id'])) {
                    triggerWebhookEvent($eventType, [
                        'campaign_id' => $data['email_campaign_id'],
                        'email' => $data['email'] ?? null,
                        'timestamp' => date('c')
                    ]);
                }
            }
        }
        
        return true;
    }
}

// API endpoint for email operations
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    try {
        // Clear any output that may have been sent
        ob_clean();
        
        require_once __DIR__ . '/../includes/csrf.php';
        requireAuth();
        header('Content-Type: application/json');
        
        $userId = $_SESSION['user_id'] ?? null;
        if (!$userId) {
            http_response_code(401);
            echo json_encode(['success' => false, 'error' => 'Unauthorized'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        }
        
        $action = $_POST['action'] ?? '';
        
        if ($action === 'create_campaign') {
            $campaignName = $_POST['campaign_name'] ?? '';
            $subjectLine = $_POST['subject_line'] ?? '';
            $emailContent = $_POST['email_content'] ?? '';
            $parentCampaignId = !empty($_POST['campaign_id']) ? intval($_POST['campaign_id']) : null;
            $recipientSource = $_POST['recipient_source'] ?? 'manual';
            $recipients = $_POST['recipients'] ?? '';
            $dailyLimit = !empty($_POST['daily_limit']) ? intval($_POST['daily_limit']) : null;
            
            if (empty($campaignName) || empty($subjectLine) || empty($emailContent)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Campaign name, subject, and content are required'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            $db = getDB();
            
            // Ensure required columns exist (for rate limiting support)
            try {
                // Check if daily_limit column exists
                $checkStmt = $db->query("SHOW COLUMNS FROM email_campaigns LIKE 'daily_limit'");
                if ($checkStmt->rowCount() === 0) {
                    // Add missing columns for rate limiting
                    $db->exec("ALTER TABLE email_campaigns ADD COLUMN daily_limit INT DEFAULT NULL");
                    $db->exec("ALTER TABLE email_campaigns ADD COLUMN daily_sent_count INT DEFAULT 0");
                    $db->exec("ALTER TABLE email_campaigns ADD COLUMN last_sent_date DATE DEFAULT NULL");
                }
            } catch (PDOException $e) {
                // Column might already exist or error occurred - continue anyway
                error_log("Column check/add error (may be safe to ignore): " . $e->getMessage());
            }
            
            // Get recipients based on source
            $recipientList = [];
            $recipientCount = 0;
            
            if ($recipientSource === 'crm') {
                // Get all subscribed CRM customers
                $stmt = $db->prepare("SELECT email, first_name, last_name FROM email_subscribers WHERE user_id = ? AND status = 'subscribed'");
                $stmt->execute([$userId]);
                $customers = $stmt->fetchAll();
                $recipientList = array_column($customers, 'email');
                $recipientCount = count($recipientList);
            } else {
                // Manual entry
                $recipientList = array_map('trim', explode(',', $recipients));
                $recipientList = array_filter($recipientList, function($email) {
                    return filter_var($email, FILTER_VALIDATE_EMAIL);
                });
                $recipientCount = count($recipientList);
            }
            
            if ($recipientCount === 0) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'No valid recipients found'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            // Determine initial status based on recipient source and automation
            $initialStatus = 'draft';
            if ($recipientSource === 'crm' && $dailyLimit && intval($dailyLimit) > 0) {
                $initialStatus = 'scheduled'; // Automatically start processing
            }
            
            // Create email campaign - check if daily_limit column exists, if not, use alternative INSERT
            try {
                $stmt = $db->prepare("
                    INSERT INTO email_campaigns 
                    (user_id, campaign_id, campaign_name, subject_line, email_content, recipient_count, daily_limit, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$userId, $parentCampaignId, $campaignName, $subjectLine, $emailContent, $recipientCount, $dailyLimit, $initialStatus]);
            } catch (PDOException $e) {
                // If daily_limit column still doesn't exist, insert without it
                if (strpos($e->getMessage(), 'daily_limit') !== false) {
                    $stmt = $db->prepare("
                        INSERT INTO email_campaigns 
                        (user_id, campaign_id, campaign_name, subject_line, email_content, recipient_count, status)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$userId, $parentCampaignId, $campaignName, $subjectLine, $emailContent, $recipientCount, $initialStatus]);
                    // If dailyLimit was set but column doesn't exist, log a warning
                    if ($dailyLimit) {
                        error_log("Warning: daily_limit feature requested but column doesn't exist. Migration 006 may need to be run.");
                    }
                } else {
                    throw $e; // Re-throw if it's a different error
                }
            }
            
            $emailCampaignId = $db->lastInsertId();
            
            // Store recipients in email_campaign_recipients table if it exists, or queue for sending
            // For CRM customers with rate limiting, we'll queue them
            if ($recipientSource === 'crm' && $dailyLimit && intval($dailyLimit) > 0) {
                // Create queue table if it doesn't exist (for safety)
                try {
                    $db->exec("
                        CREATE TABLE IF NOT EXISTS email_campaign_queue (
                            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                            email_campaign_id INT UNSIGNED NOT NULL,
                            email VARCHAR(255) NOT NULL,
                            first_name VARCHAR(100),
                            last_name VARCHAR(100),
                            status ENUM('pending', 'sending', 'sent', 'failed') DEFAULT 'pending',
                            error_message TEXT,
                            queued_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                            sent_at TIMESTAMP NULL,
                            INDEX idx_campaign_status (email_campaign_id, status),
                            INDEX idx_status_queued (status, queued_at)
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                    ");
                } catch (PDOException $e) {
                    // Table might already exist, continue
                }
                
                // Queue recipients for rate-limited sending
                try {
                    $stmt = $db->prepare("
                        INSERT INTO email_campaign_queue 
                        (email_campaign_id, email, first_name, last_name, status, queued_at)
                        SELECT ?, es.email, es.first_name, es.last_name, 'pending', NOW()
                        FROM email_subscribers es
                        WHERE es.user_id = ? AND es.status = 'subscribed'
                    ");
                    $stmt->execute([$emailCampaignId, $userId]);
                } catch (PDOException $e) {
                    error_log("Queue creation error: " . $e->getMessage());
                }
            } elseif ($recipientSource === 'manual') {
                // For manual recipients, send immediately (no rate limiting)
                try {
                    $mailer = new Mailer(); // Uses IMAP settings by default
                    $emailService = new EmailService($userId);
                    
                    $successCount = 0;
                    $failCount = 0;
                    
                    foreach ($recipientList as $recipientEmail) {
                        $result = $mailer->send([
                            'to' => [$recipientEmail],
                            'subject' => $subjectLine,
                            'html' => $emailContent
                        ]);
                        
                        if (!empty($result['success'])) {
                            $successCount++;
                        } else {
                            $failCount++;
                            error_log("Failed to send email to {$recipientEmail}: " . ($result['error'] ?? 'Unknown error'));
                        }
                    }
                    
                    // Update campaign status based on send results
                    if ($successCount > 0) {
                        $finalStatus = ($failCount === 0) ? 'sent' : 'sending';
                        $stmt = $db->prepare("
                            UPDATE email_campaigns 
                            SET status = ?, sent_count = ?, sent_at = NOW()
                            WHERE id = ?
                        ");
                        $stmt->execute([$finalStatus, $successCount, $emailCampaignId]);
                        
                        // Trigger webhook for email sent
                        triggerWebhookEvent('email.sent', [
                            'campaign_id' => $emailCampaignId,
                            'recipient_count' => $successCount,
                            'subject' => $subjectLine,
                            'sent_at' => date('c')
                        ]);
                    }
                } catch (Exception $e) {
                    error_log("Error sending manual recipient emails: " . $e->getMessage());
                    // Campaign remains in 'draft' status if sending fails
                }
            }
            
            echo json_encode([
                'success' => true,
                'campaign_id' => $emailCampaignId,
                'recipient_count' => $recipientCount,
                'message' => 'Email campaign created successfully' . ($recipientSource === 'crm' && $dailyLimit ? '. Emails will be sent at 1 per minute up to ' . $dailyLimit . ' per day.' : '')
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        } elseif ($action === 'send_test_email') {
            $to = $_POST['to'] ?? '';
            $subject = $_POST['subject'] ?? 'Test Email';
            $htmlContent = $_POST['html_content'] ?? '';
            
            if (empty($to) || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Valid email address required'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            if (empty($htmlContent)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Email content required'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            try {
                $emailService = new EmailService($userId);
                
                // Send test email using SMTP (test mode)
                $mailer = new Mailer();
                $result = $mailer->send([
                    'to' => $to,
                    'subject' => $subject . ' [TEST]',
                    'html' => $htmlContent
                ]);
                
                if (!empty($result['success'])) {
                    echo json_encode([
                        'success' => true,
                        'message' => 'Test email sent successfully to ' . $to
                    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                } else {
                    http_response_code(500);
                    echo json_encode([
                        'success' => false,
                        'error' => $result['error'] ?? 'Failed to send test email'
                    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                }
            } catch (Exception $e) {
                http_response_code(500);
                error_log('Test email error: ' . $e->getMessage());
                ob_clean();
                echo json_encode([
                    'success' => false,
                    'error' => 'Error: ' . $e->getMessage()
                ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            }
            exit;
        } elseif ($action === 'test_imap_email') {
            // Test IMAP email sending with provided settings
            $testEmail = $_POST['test_email'] ?? '';
            $testSubject = $_POST['test_subject'] ?? 'Test Email';
            $testMessage = $_POST['test_message'] ?? '';
            
            // Get IMAP settings from form or use defaults
            $imapHost = $_POST['imap_host'] ?? (defined('IMAP_HOST') ? IMAP_HOST : '');
            $imapPort = $_POST['imap_port'] ?? (defined('IMAP_PORT') ? IMAP_PORT : 993);
            $imapUser = $_POST['imap_user'] ?? (defined('IMAP_USER') ? IMAP_USER : '');
            $imapPass = $_POST['imap_pass'] ?? (defined('IMAP_PASS') ? IMAP_PASS : '');
            $imapEncryption = $_POST['imap_encryption'] ?? (defined('IMAP_ENCRYPTION') ? IMAP_ENCRYPTION : 'ssl');
            
            if (empty($testEmail) || !filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Valid test email address required'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            if (empty($imapHost) || empty($imapUser) || empty($imapPass)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'IMAP settings are required. Please configure IMAP settings first.'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            try {
                // Create Mailer with IMAP settings override
                $overrides = [
                    'host' => preg_replace('/^imaps?\./', 'smtp.', $imapHost),
                    'port' => (int)$imapPort,
                    'username' => $imapUser,
                    'password' => $imapPass,
                    'secure' => $imapEncryption === 'ssl' ? 'ssl' : ($imapEncryption === 'tls' ? 'tls' : ''),
                    'from_email' => $imapUser,
                    'from_name' => 'Marketing Platform Test'
                ];
                
                $mailer = new Mailer($overrides);
                $result = $mailer->send([
                    'to' => $testEmail,
                    'subject' => $testSubject . ' [TEST]',
                    'html' => nl2br(htmlspecialchars($testMessage))
                ]);
                
                if (!empty($result['success'])) {
                    echo json_encode([
                        'success' => true,
                        'message' => 'Test email sent successfully to ' . $testEmail
                    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                } else {
                    http_response_code(500);
                    echo json_encode([
                        'success' => false,
                        'error' => $result['error'] ?? 'Failed to send test email'
                    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                }
            } catch (Exception $e) {
                http_response_code(500);
                error_log('Test IMAP email error: ' . $e->getMessage());
                ob_clean();
                echo json_encode([
                    'success' => false,
                    'error' => 'Error: ' . $e->getMessage()
                ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            }
            exit;
        } elseif ($action === 'delete_campaign') {
            $campaignId = !empty($_POST['campaign_id']) ? intval($_POST['campaign_id']) : null;
            
            if (!$campaignId) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Campaign ID is required'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            $db = getDB();
            
            // Verify ownership
            $stmt = $db->prepare("SELECT user_id FROM email_campaigns WHERE id = ?");
            $stmt->execute([$campaignId]);
            $campaign = $stmt->fetch();
            
            if (!$campaign) {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Campaign not found'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            if ($campaign['user_id'] != $userId) {
                http_response_code(403);
                echo json_encode(['success' => false, 'error' => 'Unauthorized'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            // Delete campaign queue items first
            try {
                $stmt = $db->prepare("DELETE FROM email_campaign_queue WHERE email_campaign_id = ?");
                $stmt->execute([$campaignId]);
            } catch (PDOException $e) {
                // Queue table might not exist, continue
            }
            
            // Delete campaign
            $stmt = $db->prepare("DELETE FROM email_campaigns WHERE id = ?");
            $stmt->execute([$campaignId]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Campaign deleted successfully'
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        } elseif ($action === 'get_campaign') {
            $campaignId = !empty($_GET['campaign_id']) ? intval($_GET['campaign_id']) : null;
            
            if (!$campaignId) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Campaign ID is required'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            $db = getDB();
            
            $stmt = $db->prepare("SELECT id, campaign_name, subject_line, email_content, recipient_count, sent_count, status FROM email_campaigns WHERE id = ? AND user_id = ?");
            $stmt->execute([$campaignId, $userId]);
            $campaign = $stmt->fetch();
            
            if (!$campaign) {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Campaign not found'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            echo json_encode([
                'success' => true,
                'campaign' => $campaign
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        } elseif ($action === 'send_ad_individually') {
            $campaignId = !empty($_POST['campaign_id']) ? intval($_POST['campaign_id']) : null;
            $recipientEmail = $_POST['recipient_email'] ?? '';
            $subject = $_POST['subject'] ?? '';
            $additionalMessage = $_POST['additional_message'] ?? '';
            
            if (!$campaignId || empty($recipientEmail) || !filter_var($recipientEmail, FILTER_VALIDATE_EMAIL)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Valid campaign ID and recipient email are required'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            $db = getDB();
            
            // Get campaign
            $stmt = $db->prepare("SELECT email_content, subject_line FROM email_campaigns WHERE id = ? AND user_id = ?");
            $stmt->execute([$campaignId, $userId]);
            $campaign = $stmt->fetch();
            
            if (!$campaign) {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Campaign not found'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                exit;
            }
            
            // Build email content
            $emailSubject = !empty($subject) ? $subject : $campaign['subject_line'];
            $emailContent = '';
            
            if (!empty($additionalMessage)) {
                $emailContent .= '<p>' . nl2br(htmlspecialchars($additionalMessage)) . '</p>';
            }
            
            // Extract ad HTML from campaign content (look for ad HTML or use full content)
            $emailContent .= $campaign['email_content'];
            
            try {
                $mailer = new Mailer(); // Uses IMAP settings by default
                $result = $mailer->send([
                    'to' => [$recipientEmail],
                    'subject' => $emailSubject,
                    'html' => $emailContent
                ]);
                
                if (!empty($result['success'])) {
                    echo json_encode([
                        'success' => true,
                        'message' => 'Ad sent successfully to ' . $recipientEmail
                    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                } else {
                    http_response_code(500);
                    echo json_encode([
                        'success' => false,
                        'error' => $result['error'] ?? 'Failed to send ad'
                    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                }
            } catch (Exception $e) {
                http_response_code(500);
                error_log('Send ad individually error: ' . $e->getMessage());
                ob_clean();
                echo json_encode([
                    'success' => false,
                    'error' => 'Error: ' . $e->getMessage()
                ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            }
            exit;
        } else {
            http_response_code(400);
            ob_clean();
            echo json_encode(['success' => false, 'error' => 'Invalid action'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        }
    } catch (Exception $e) {
        http_response_code(500);
        error_log('Email Service API Error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString());
        ob_clean();
        echo json_encode(['success' => false, 'error' => 'Server error: ' . $e->getMessage()], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    } catch (Throwable $e) {
        http_response_code(500);
        error_log('Email Service API Fatal Error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString());
        ob_clean();
        echo json_encode(['success' => false, 'error' => 'Fatal error: ' . $e->getMessage()], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }
}

// Webhook endpoint for email tracking
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['webhook'])) {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    $emailService = new EmailService();
    $emailService->handleWebhook($data);
    
    http_response_code(200);
    echo json_encode(['success' => true]);
}

