<?php
/**
 * Email Segmentation API
 * Handles subscriber segmentation and list management
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');
requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if ($action === 'segments') {
                $stmt = $db->prepare("SELECT * FROM email_segments WHERE user_id = ? ORDER BY created_at DESC");
                $stmt->execute([$userId]);
                $segments = $stmt->fetchAll();
                
                // Get subscriber counts for each segment
                foreach ($segments as &$segment) {
                    $segment['segment_rules'] = json_decode($segment['segment_rules'], true);
                    
                    // Count subscribers in segment
                    $stmt2 = $db->prepare("
                        SELECT COUNT(*) FROM email_segment_subscribers WHERE segment_id = ?
                    ");
                    $stmt2->execute([$segment['id']]);
                    $segment['subscriber_count'] = $stmt2->fetchColumn();
                }
                
                echo json_encode(['success' => true, 'data' => $segments]);
            } elseif ($action === 'segment_subscribers') {
                $segmentId = intval($_GET['segment_id'] ?? 0);
                
                $stmt = $db->prepare("
                    SELECT es.* FROM email_subscribers es
                    JOIN email_segment_subscribers ess ON es.id = ess.subscriber_id
                    WHERE ess.segment_id = ? AND es.user_id = ?
                    ORDER BY es.subscribed_at DESC
                ");
                $stmt->execute([$segmentId, $userId]);
                $subscribers = $stmt->fetchAll();
                
                echo json_encode(['success' => true, 'data' => $subscribers]);
            }
            break;
            
        case 'POST':
            requireCSRFToken();
            
            if ($action === 'create_segment') {
                $segmentName = trim($_POST['segment_name'] ?? '');
                $segmentRules = $_POST['segment_rules'] ?? '{}';
                
                if (empty($segmentName)) {
                    throw new Exception('Segment name is required');
                }
                
                // Validate JSON
                $decodedRules = json_decode($segmentRules, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('Invalid segment rules JSON');
                }
                
                $stmt = $db->prepare("
                    INSERT INTO email_segments (user_id, segment_name, segment_rules)
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$userId, $segmentName, $segmentRules]);
                $segmentId = $db->lastInsertId();
                
                // Apply rules to populate segment
                applySegmentRules($db, $segmentId, $userId, $decodedRules);
                
                echo json_encode(['success' => true, 'segment_id' => $segmentId]);
                
            } elseif ($action === 'update_segment') {
                $segmentId = intval($_POST['segment_id'] ?? 0);
                $segmentName = trim($_POST['segment_name'] ?? '');
                $segmentRules = $_POST['segment_rules'] ?? '{}';
                
                $stmt = $db->prepare("SELECT user_id FROM email_segments WHERE id = ?");
                $stmt->execute([$segmentId]);
                $segment = $stmt->fetch();
                
                if (!$segment || $segment['user_id'] != $userId) {
                    throw new Exception('Segment not found');
                }
                
                $decodedRules = json_decode($segmentRules, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('Invalid segment rules JSON');
                }
                
                $stmt = $db->prepare("
                    UPDATE email_segments 
                    SET segment_name = ?, segment_rules = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$segmentName, $segmentRules, $segmentId]);
                
                // Clear and repopulate segment
                $stmt = $db->prepare("DELETE FROM email_segment_subscribers WHERE segment_id = ?");
                $stmt->execute([$segmentId]);
                applySegmentRules($db, $segmentId, $userId, $decodedRules);
                
                echo json_encode(['success' => true]);
            }
            break;
            
        case 'DELETE':
            requireCSRFToken();
            
            if ($action === 'delete_segment') {
                $segmentId = intval($_GET['id'] ?? $_POST['id'] ?? 0);
                
                $stmt = $db->prepare("SELECT user_id FROM email_segments WHERE id = ?");
                $stmt->execute([$segmentId]);
                $segment = $stmt->fetch();
                
                if (!$segment || $segment['user_id'] != $userId) {
                    throw new Exception('Segment not found or access denied');
                }
                
                $stmt = $db->prepare("DELETE FROM email_segments WHERE id = ?");
                $stmt->execute([$segmentId]);
                
                echo json_encode(['success' => true, 'message' => 'Segment deleted']);
            }
            break;
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

// Helper function to apply segment rules
function applySegmentRules($db, $segmentId, $userId, $rules) {
    $query = "SELECT id FROM email_subscribers WHERE user_id = ? AND status = 'subscribed'";
    $params = [$userId];
    
    // Build query based on rules
    if (isset($rules['email_contains'])) {
        $query .= " AND email LIKE ?";
        $params[] = '%' . $rules['email_contains'] . '%';
    }
    
    if (isset($rules['subscribed_after'])) {
        $query .= " AND subscribed_at >= ?";
        $params[] = $rules['subscribed_after'];
    }
    
    if (isset($rules['subscribed_before'])) {
        $query .= " AND subscribed_at <= ?";
        $params[] = $rules['subscribed_before'];
    }
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $subscribers = $stmt->fetchAll();
    
    // Add subscribers to segment
    $insertStmt = $db->prepare("INSERT INTO email_segment_subscribers (segment_id, subscriber_id) VALUES (?, ?)");
    foreach ($subscribers as $subscriber) {
        try {
            $insertStmt->execute([$segmentId, $subscriber['id']]);
        } catch (PDOException $e) {
            // Skip duplicates
        }
    }
    
    // Update segment count
    $countStmt = $db->prepare("UPDATE email_segments SET subscriber_count = (SELECT COUNT(*) FROM email_segment_subscribers WHERE segment_id = ?) WHERE id = ?");
    $countStmt->execute([$segmentId, $segmentId]);
}

