<?php
/**
 * Email Campaign Scheduling API
 * Handles scheduling of email campaigns for future delivery
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');
requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if ($action === 'scheduled') {
                // Get all scheduled campaigns
                $stmt = $db->prepare("
                    SELECT * FROM email_campaigns 
                    WHERE user_id = ? AND status IN ('scheduled', 'sending')
                    ORDER BY scheduled_at ASC
                ");
                $stmt->execute([$userId]);
                $campaigns = $stmt->fetchAll();
                
                echo json_encode(['success' => true, 'data' => $campaigns]);
            }
            break;
            
        case 'POST':
            requireCSRFToken();
            
            if ($action === 'schedule') {
                $campaignId = intval($_POST['campaign_id'] ?? 0);
                $scheduledAt = $_POST['scheduled_at'] ?? '';
                
                if (!$campaignId || empty($scheduledAt)) {
                    throw new Exception('Campaign ID and scheduled time are required');
                }
                
                // Validate scheduled time is in the future
                $scheduledDateTime = new DateTime($scheduledAt);
                $now = new DateTime();
                if ($scheduledDateTime <= $now) {
                    throw new Exception('Scheduled time must be in the future');
                }
                
                // Verify campaign ownership
                $stmt = $db->prepare("SELECT user_id, status FROM email_campaigns WHERE id = ?");
                $stmt->execute([$campaignId]);
                $campaign = $stmt->fetch();
                
                if (!$campaign || $campaign['user_id'] != $userId) {
                    throw new Exception('Campaign not found or access denied');
                }
                
                if ($campaign['status'] !== 'draft') {
                    throw new Exception('Only draft campaigns can be scheduled');
                }
                
                // Update campaign status and scheduled time
                $stmt = $db->prepare("
                    UPDATE email_campaigns 
                    SET status = 'scheduled', scheduled_at = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$scheduledAt, $campaignId]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Campaign scheduled successfully',
                    'scheduled_at' => $scheduledAt
                ]);
                
            } elseif ($action === 'cancel_schedule') {
                $campaignId = intval($_POST['campaign_id'] ?? 0);
                
                // Verify campaign ownership
                $stmt = $db->prepare("SELECT user_id, status FROM email_campaigns WHERE id = ?");
                $stmt->execute([$campaignId]);
                $campaign = $stmt->fetch();
                
                if (!$campaign || $campaign['user_id'] != $userId) {
                    throw new Exception('Campaign not found or access denied');
                }
                
                if (!in_array($campaign['status'], ['scheduled', 'sending'])) {
                    throw new Exception('Campaign is not scheduled');
                }
                
                // Cancel scheduling
                $stmt = $db->prepare("
                    UPDATE email_campaigns 
                    SET status = 'draft', scheduled_at = NULL, updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$campaignId]);
                
                echo json_encode(['success' => true, 'message' => 'Schedule cancelled']);
            }
            break;
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

