<?php
/**
 * Email Queue Processor
 * Processes queued emails with rate limiting (1 per minute, respecting daily limits)
 * Run via cron: php market/api/email_queue_processor.php
 * Or call via HTTP: GET market/api/email_queue_processor.php
 */

require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../config/email_config.php';
require_once __DIR__ . '/../includes/Mailer.php';

header('Content-Type: application/json');

$db = getDB();

// Get campaigns that need processing (with daily limits and pending queue items)
$stmt = $db->prepare("
    SELECT DISTINCT ec.id, ec.user_id, ec.campaign_name, ec.subject_line, ec.email_content, 
           ec.daily_limit, ec.daily_sent_count, ec.last_sent_date
    FROM email_campaigns ec
    INNER JOIN email_campaign_queue ecq ON ec.id = ecq.email_campaign_id
    WHERE ec.status IN ('draft', 'scheduled', 'sending')
      AND ecq.status = 'pending'
      AND ec.daily_limit IS NOT NULL
    ORDER BY ec.id ASC
    LIMIT 1
");
$stmt->execute();
$campaign = $stmt->fetch();

if (!$campaign) {
    echo json_encode(['success' => true, 'message' => 'No campaigns to process']);
    exit;
}

// Check daily limit - reset if new day
$today = date('Y-m-d');
if ($campaign['last_sent_date'] !== $today) {
    // Reset daily count for new day
    $stmt = $db->prepare("UPDATE email_campaigns SET daily_sent_count = 0, last_sent_date = ? WHERE id = ?");
    $stmt->execute([$today, $campaign['id']]);
    $campaign['daily_sent_count'] = 0;
}

// Check if we've reached daily limit
if ($campaign['daily_sent_count'] >= $campaign['daily_limit']) {
    echo json_encode([
        'success' => true, 
        'message' => "Daily limit reached for campaign {$campaign['id']}. {$campaign['daily_sent_count']}/{$campaign['daily_limit']} sent today."
    ]);
    exit;
}

// Get next pending email from queue
$stmt = $db->prepare("
    SELECT id, email, first_name, last_name
    FROM email_campaign_queue
    WHERE email_campaign_id = ? AND status = 'pending'
    ORDER BY queued_at ASC
    LIMIT 1
");
$stmt->execute([$campaign['id']]);
$queueItem = $stmt->fetch();

if (!$queueItem) {
    // No more pending items, mark campaign as sent
    $stmt = $db->prepare("UPDATE email_campaigns SET status = 'sent' WHERE id = ?");
    $stmt->execute([$campaign['id']]);
    echo json_encode(['success' => true, 'message' => 'All emails sent for campaign']);
    exit;
}

// Update queue item to 'sending'
$stmt = $db->prepare("UPDATE email_campaign_queue SET status = 'sending' WHERE id = ?");
$stmt->execute([$queueItem['id']]);

// Send email
try {
    $mailer = new Mailer(); // Uses IMAP settings by default
    $recipientName = trim(($queueItem['first_name'] ?? '') . ' ' . ($queueItem['last_name'] ?? ''));
    $to = $recipientName ? "{$recipientName} <{$queueItem['email']}>" : $queueItem['email'];
    
    $result = $mailer->send([
        'to' => [$to],
        'subject' => $campaign['subject_line'],
        'html' => $campaign['email_content']
    ]);
    
    if (!empty($result['success'])) {
        // Mark as sent
        $stmt = $db->prepare("
            UPDATE email_campaign_queue 
            SET status = 'sent', sent_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$queueItem['id']]);
        
        // Update campaign stats
        $newDailyCount = $campaign['daily_sent_count'] + 1;
        $stmt = $db->prepare("
            UPDATE email_campaigns 
            SET sent_count = sent_count + 1,
                daily_sent_count = ?,
                last_sent_date = ?,
                status = CASE 
                    WHEN (SELECT COUNT(*) FROM email_campaign_queue WHERE email_campaign_id = ? AND status = 'pending') > 0 
                    THEN 'sending' 
                    ELSE 'sent' 
                END
            WHERE id = ?
        ");
        $stmt->execute([$newDailyCount, $today, $campaign['id'], $campaign['id']]);
        
        echo json_encode([
            'success' => true,
            'message' => "Email sent to {$queueItem['email']}. Daily progress: {$newDailyCount}/{$campaign['daily_limit']}"
        ]);
    } else {
        // Mark as failed
        $stmt = $db->prepare("
            UPDATE email_campaign_queue 
            SET status = 'failed', error_message = ? 
            WHERE id = ?
        ");
        $errorMsg = $result['error'] ?? 'Unknown error';
        $stmt->execute([substr($errorMsg, 0, 255), $queueItem['id']]);
        
        echo json_encode([
            'success' => false,
            'error' => "Failed to send email to {$queueItem['email']}: {$errorMsg}"
        ]);
    }
} catch (Exception $e) {
    // Mark as failed
    $stmt = $db->prepare("
        UPDATE email_campaign_queue 
        SET status = 'failed', error_message = ? 
        WHERE id = ?
    ");
    $stmt->execute([substr($e->getMessage(), 0, 255), $queueItem['id']]);
    
    echo json_encode([
        'success' => false,
        'error' => "Exception: " . $e->getMessage()
    ]);
}

