<?php
/**
 * Email Campaign CSV Import
 * Accepts CSV with headers: email[,name]
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');

requireClient();
$userId = $_SESSION['user_id'] ?? null;

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

if (!isset($_POST['campaign_id']) || empty($_FILES['file']['tmp_name'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'campaign_id and CSV file are required']);
    exit;
}

$campaignId = (int)$_POST['campaign_id'];

// Basic campaign ownership check
$db = getDB();
$stmt = $db->prepare("SELECT id FROM email_campaigns WHERE id = ? AND user_id = ? LIMIT 1");
$stmt->execute([$campaignId, $userId]);
$campaign = $stmt->fetch();
if (!$campaign) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Campaign not found']);
    exit;
}

// Ensure recipients table exists
$db->exec("CREATE TABLE IF NOT EXISTS email_campaign_recipients (
    id INT AUTO_INCREMENT PRIMARY KEY,
    campaign_id INT NOT NULL,
    email VARCHAR(255) NOT NULL,
    name VARCHAR(255) NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY uniq_campaign_email (campaign_id, email),
    INDEX idx_campaign_id (campaign_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

$filePath = $_FILES['file']['tmp_name'];
$handle = fopen($filePath, 'r');
if ($handle === false) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Unable to read uploaded file']);
    exit;
}

$header = fgetcsv($handle);
if ($header === false) {
    echo json_encode(['success' => false, 'message' => 'Empty CSV']);
    fclose($handle);
    exit;
}

// Normalize headers
$normalized = array_map(function($h){ return strtolower(trim($h)); }, $header);
$emailIndex = array_search('email', $normalized);
$nameIndex = array_search('name', $normalized);

if ($emailIndex === false) {
    echo json_encode(['success' => false, 'message' => 'CSV must include an email column']);
    fclose($handle);
    exit;
}

$inserted = 0;
$updatedCount = 0;
$invalid = 0;

$db->beginTransaction();
$insertStmt = $db->prepare("INSERT IGNORE INTO email_campaign_recipients (campaign_id, email, name) VALUES (?, ?, ?)");

while (($row = fgetcsv($handle)) !== false) {
    if (!isset($row[$emailIndex])) { $invalid++; continue; }
    $email = trim($row[$emailIndex]);
    if (filter_var($email, FILTER_VALIDATE_EMAIL) === false) { $invalid++; continue; }
    $name = ($nameIndex !== false && isset($row[$nameIndex])) ? trim($row[$nameIndex]) : null;
    $insertStmt->execute([$campaignId, $email, $name]);
    if ($insertStmt->rowCount() > 0) { $inserted++; }
}

fclose($handle);

// Update campaign recipient_count
$countStmt = $db->prepare("SELECT COUNT(*) FROM email_campaign_recipients WHERE campaign_id = ?");
$countStmt->execute([$campaignId]);
$recipientCount = (int)$countStmt->fetchColumn();

$updStmt = $db->prepare("UPDATE email_campaigns SET recipient_count = ? WHERE id = ?");
$updStmt->execute([$recipientCount, $campaignId]);

$db->commit();

echo json_encode([
    'success' => true,
    'inserted' => $inserted,
    'invalid' => $invalid,
    'recipient_count' => $recipientCount
]);


