<?php
/**
 * Discount Code Analytics API
 * Provides comprehensive analytics and reporting for discount codes
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');
requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

$action = $_GET['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

try {
    if ($method === 'GET') {
        switch ($action) {
            case 'overview':
                // Get overview stats for all discount codes
                $stmt = $db->prepare("
                    SELECT 
                        COUNT(*) as total_codes,
                        SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_codes,
                        SUM(usage_count) as total_usages,
                        SUM(CASE 
                            WHEN discount_type = 'percentage' THEN usage_count * discount_value / 100
                            ELSE usage_count * discount_value 
                        END) as total_discount_amount
                    FROM discount_codes
                    WHERE user_id = ?
                ");
                $stmt->execute([$userId]);
                $overview = $stmt->fetch();
                
                echo json_encode(['success' => true, 'data' => $overview]);
                break;
                
            case 'code_stats':
                // Get detailed stats for a specific discount code
                $codeId = intval($_GET['code_id'] ?? 0);
                
                if (!$codeId) {
                    throw new Exception('Code ID is required');
                }
                
                // Get code info
                $stmt = $db->prepare("SELECT * FROM discount_codes WHERE id = ? AND user_id = ?");
                $stmt->execute([$codeId, $userId]);
                $code = $stmt->fetch();
                
                if (!$code) {
                    throw new Exception('Discount code not found');
                }
                
                // Get usage statistics
                $stmt = $db->prepare("
                    SELECT 
                        COUNT(*) as total_usages,
                        SUM(discount_amount) as total_discount_given,
                        SUM(conversion_value) as total_revenue,
                        AVG(order_value) as avg_order_value,
                        MIN(used_at) as first_used,
                        MAX(used_at) as last_used
                    FROM discount_code_usage
                    WHERE discount_code_id = ?
                ");
                $stmt->execute([$codeId]);
                $stats = $stmt->fetch();
                
                // Get usage by date
                $stmt = $db->prepare("
                    SELECT DATE(used_at) as usage_date, COUNT(*) as usage_count, 
                           SUM(discount_amount) as discount_amount,
                           SUM(conversion_value) as revenue
                    FROM discount_code_usage
                    WHERE discount_code_id = ?
                    GROUP BY DATE(used_at)
                    ORDER BY usage_date DESC
                    LIMIT 30
                ");
                $stmt->execute([$codeId]);
                $usageByDate = $stmt->fetchAll();
                
                echo json_encode([
                    'success' => true,
                    'data' => [
                        'code' => $code,
                        'stats' => $stats,
                        'usage_by_date' => $usageByDate
                    ]
                ]);
                break;
                
            case 'usage_tracking':
                // Get detailed usage tracking
                $codeId = isset($_GET['code_id']) ? intval($_GET['code_id']) : null;
                $startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
                $endDate = $_GET['end_date'] ?? date('Y-m-d');
                
                $query = "
                    SELECT dcu.*, dc.code, dc.discount_type, dc.discount_value
                    FROM discount_code_usage dcu
                    JOIN discount_codes dc ON dcu.discount_code_id = dc.id
                    WHERE dc.user_id = ? 
                    AND DATE(dcu.used_at) BETWEEN ? AND ?
                ";
                $params = [$userId, $startDate, $endDate];
                
                if ($codeId) {
                    $query .= " AND dcu.discount_code_id = ?";
                    $params[] = $codeId;
                }
                
                $query .= " ORDER BY dcu.used_at DESC LIMIT 100";
                
                $stmt = $db->prepare($query);
                $stmt->execute($params);
                $usage = $stmt->fetchAll();
                
                echo json_encode(['success' => true, 'data' => $usage]);
                break;
                
            case 'expiring_soon':
                // Get codes expiring soon
                $days = intval($_GET['days'] ?? 30);
                
                $stmt = $db->prepare("
                    SELECT * FROM discount_codes
                    WHERE user_id = ? 
                    AND status = 'active'
                    AND end_date IS NOT NULL
                    AND end_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL ? DAY)
                    ORDER BY end_date ASC
                ");
                $stmt->execute([$userId, $days]);
                $expiring = $stmt->fetchAll();
                
                echo json_encode(['success' => true, 'data' => $expiring]);
                break;
                
            case 'reports':
                // Generate comprehensive reports
                $reportType = $_GET['report_type'] ?? 'summary';
                $startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
                $endDate = $_GET['end_date'] ?? date('Y-m-d');
                
                switch ($reportType) {
                    case 'summary':
                        $stmt = $db->prepare("
                            SELECT 
                                dc.id,
                                dc.code,
                                dc.discount_type,
                                dc.discount_value,
                                dc.status,
                                dc.usage_count,
                                dc.start_date,
                                dc.end_date,
                                COALESCE(SUM(dcu.discount_amount), 0) as total_discount_given,
                                COALESCE(SUM(dcu.conversion_value), 0) as total_revenue
                            FROM discount_codes dc
                            LEFT JOIN discount_code_usage dcu ON dc.id = dcu.discount_code_id 
                                AND DATE(dcu.used_at) BETWEEN ? AND ?
                            WHERE dc.user_id = ?
                            GROUP BY dc.id
                            ORDER BY dc.created_at DESC
                        ");
                        $stmt->execute([$startDate, $endDate, $userId]);
                        $report = $stmt->fetchAll();
                        break;
                        
                    default:
                        throw new Exception('Invalid report type');
                }
                
                echo json_encode(['success' => true, 'data' => $report]);
                break;
        }
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

