<?php
/**
 * Comprehensive Analytics API
 * Provides detailed metrics tracking, charts data, ROI calculations, and conversion tracking
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');
requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

$action = $_GET['action'] ?? '';
$startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$endDate = $_GET['end_date'] ?? date('Y-m-d');

try {
    switch ($action) {
        case 'dashboard_metrics':
            // Get comprehensive dashboard metrics
            $metrics = [];
            
            // Campaign metrics
            $stmt = $db->prepare("
                SELECT 
                    COUNT(*) as total_campaigns,
                    SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_campaigns,
                    SUM(budget) as total_budget,
                    COUNT(DISTINCT campaign_type) as campaign_types
                FROM campaigns
                WHERE user_id = ?
            ");
            $stmt->execute([$userId]);
            $metrics['campaigns'] = $stmt->fetch();
            
            // Ad performance metrics
            $stmt = $db->prepare("
                SELECT 
                    SUM(impressions) as total_impressions,
                    SUM(clicks) as total_clicks,
                    SUM(conversions) as total_conversions,
                    SUM(cost) as total_cost,
                    SUM(revenue) as total_revenue,
                    CASE 
                        WHEN SUM(clicks) > 0 THEN (SUM(cost) / SUM(clicks))
                        ELSE 0 
                    END as avg_cpc,
                    CASE 
                        WHEN SUM(impressions) > 0 THEN (SUM(clicks) / SUM(impressions) * 100)
                        ELSE 0 
                    END as ctr,
                    CASE 
                        WHEN SUM(clicks) > 0 THEN (SUM(conversions) / SUM(clicks) * 100)
                        ELSE 0 
                    END as conversion_rate,
                    CASE 
                        WHEN SUM(cost) > 0 THEN ((SUM(revenue) - SUM(cost)) / SUM(cost) * 100)
                        ELSE 0 
                    END as roi
                FROM ad_performance
                WHERE user_id = ? AND date BETWEEN ? AND ?
            ");
            $stmt->execute([$userId, $startDate, $endDate]);
            $metrics['ad_performance'] = $stmt->fetch();
            
            // Email metrics
            $stmt = $db->prepare("
                SELECT 
                    COUNT(*) as total_campaigns,
                    SUM(sent_count) as total_sent,
                    SUM(opened_count) as total_opened,
                    SUM(clicked_count) as total_clicked,
                    CASE 
                        WHEN SUM(sent_count) > 0 THEN (SUM(opened_count) / SUM(sent_count) * 100)
                        ELSE 0 
                    END as open_rate,
                    CASE 
                        WHEN SUM(sent_count) > 0 THEN (SUM(clicked_count) / SUM(sent_count) * 100)
                        ELSE 0 
                    END as click_rate
                FROM email_campaigns
                WHERE user_id = ? AND sent_at BETWEEN ? AND ?
            ");
            $stmt->execute([$userId, $startDate, $endDate]);
            $metrics['email'] = $stmt->fetch();
            
            // User metrics
            $stmt = $db->prepare("
                SELECT 
                    COUNT(*) as total_visits,
                    COUNT(DISTINCT session_id) as unique_visitors,
                    COUNT(CASE WHEN conversion_type IS NOT NULL THEN 1 END) as conversions,
                    SUM(conversion_value) as total_conversion_value
                FROM user_metrics
                WHERE user_id = ? AND visited_at BETWEEN ? AND ?
            ");
            $stmt->execute([$userId, $startDate, $endDate]);
            $metrics['user_metrics'] = $stmt->fetch();
            
            echo json_encode(['success' => true, 'data' => $metrics]);
            break;
            
        case 'time_series':
            // Get time series data for charts
            $metricType = $_GET['metric_type'] ?? 'revenue';
            $groupBy = $_GET['group_by'] ?? 'day'; // day, week, month
            
            switch ($metricType) {
                case 'revenue':
                    $interval = $groupBy === 'day' ? 'DATE' : ($groupBy === 'week' ? 'YEARWEEK' : 'YEAR_MONTH');
                    $stmt = $db->prepare("
                        SELECT 
                            DATE(date) as period,
                            SUM(revenue) as value
                        FROM ad_performance
                        WHERE user_id = ? AND date BETWEEN ? AND ?
                        GROUP BY DATE(date)
                        ORDER BY period ASC
                    ");
                    $stmt->execute([$userId, $startDate, $endDate]);
                    break;
                    
                case 'cost':
                    $stmt = $db->prepare("
                        SELECT 
                            DATE(date) as period,
                            SUM(cost) as value
                        FROM ad_performance
                        WHERE user_id = ? AND date BETWEEN ? AND ?
                        GROUP BY DATE(date)
                        ORDER BY period ASC
                    ");
                    $stmt->execute([$userId, $startDate, $endDate]);
                    break;
                    
                case 'conversions':
                    $stmt = $db->prepare("
                        SELECT 
                            DATE(visited_at) as period,
                            COUNT(*) as value
                        FROM user_metrics
                        WHERE user_id = ? AND conversion_type IS NOT NULL 
                        AND visited_at BETWEEN ? AND ?
                        GROUP BY DATE(visited_at)
                        ORDER BY period ASC
                    ");
                    $stmt->execute([$userId, $startDate, $endDate]);
                    break;
                    
                default:
                    throw new Exception('Invalid metric type');
            }
            
            $data = $stmt->fetchAll();
            echo json_encode(['success' => true, 'data' => $data]);
            break;
            
        case 'conversion_tracking':
            // Get conversion tracking data
            $stmt = $db->prepare("
                SELECT 
                    conversion_type,
                    COUNT(*) as count,
                    SUM(conversion_value) as total_value,
                    AVG(conversion_value) as avg_value,
                    MIN(conversion_value) as min_value,
                    MAX(conversion_value) as max_value
                FROM user_metrics
                WHERE user_id = ? 
                AND conversion_type IS NOT NULL
                AND visited_at BETWEEN ? AND ?
                GROUP BY conversion_type
                ORDER BY count DESC
            ");
            $stmt->execute([$userId, $startDate, $endDate]);
            $conversions = $stmt->fetchAll();
            
            echo json_encode(['success' => true, 'data' => $conversions]);
            break;
            
        case 'roi_analysis':
            // Calculate ROI for campaigns
            $stmt = $db->prepare("
                SELECT 
                    c.id,
                    c.campaign_name,
                    c.campaign_type,
                    c.budget,
                    COALESCE(SUM(ap.cost), 0) as total_cost,
                    COALESCE(SUM(ap.revenue), 0) as total_revenue,
                    COALESCE(SUM(ap.conversions), 0) as total_conversions,
                    CASE 
                        WHEN SUM(ap.cost) > 0 THEN ((SUM(ap.revenue) - SUM(ap.cost)) / SUM(ap.cost) * 100)
                        ELSE 0 
                    END as roi_percentage,
                    (COALESCE(SUM(ap.revenue), 0) - COALESCE(SUM(ap.cost), 0)) as net_profit
                FROM campaigns c
                LEFT JOIN ad_performance ap ON c.id = ap.campaign_id 
                    AND ap.date BETWEEN ? AND ?
                WHERE c.user_id = ?
                GROUP BY c.id
                ORDER BY roi_percentage DESC
            ");
            $stmt->execute([$startDate, $endDate, $userId]);
            $roi = $stmt->fetchAll();
            
            echo json_encode(['success' => true, 'data' => $roi]);
            break;
            
        case 'channel_performance':
            // Compare performance across channels
            $stmt = $db->prepare("
                SELECT 
                    campaign_type as channel,
                    COUNT(DISTINCT c.id) as campaign_count,
                    SUM(ap.impressions) as impressions,
                    SUM(ap.clicks) as clicks,
                    SUM(ap.conversions) as conversions,
                    SUM(ap.cost) as cost,
                    SUM(ap.revenue) as revenue,
                    CASE 
                        WHEN SUM(ap.clicks) > 0 THEN (SUM(ap.cost) / SUM(ap.clicks))
                        ELSE 0 
                    END as cpc,
                    CASE 
                        WHEN SUM(ap.cost) > 0 THEN ((SUM(ap.revenue) - SUM(ap.cost)) / SUM(ap.cost) * 100)
                        ELSE 0 
                    END as roi
                FROM campaigns c
                LEFT JOIN ad_performance ap ON c.id = ap.campaign_id 
                    AND ap.date BETWEEN ? AND ?
                WHERE c.user_id = ?
                GROUP BY campaign_type
            ");
            $stmt->execute([$startDate, $endDate, $userId]);
            $channels = $stmt->fetchAll();
            
            echo json_encode(['success' => true, 'data' => $channels]);
            break;
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

