<?php
/**
 * Campaign Templates API
 * Handles campaign template creation, management, and usage
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');

requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

// Get action from request
$action = $_GET['action'] ?? $_POST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if ($action === 'list') {
                // List templates (public and user's own)
                $includePublic = isset($_GET['include_public']) ? (bool)$_GET['include_public'] : true;
                $templateType = $_GET['template_type'] ?? null;
                
                $query = "SELECT * FROM campaign_templates 
                          WHERE (user_id = ? OR (is_public = 1 AND ?))";
                $params = [$userId, $includePublic ? 1 : 0];
                
                if ($templateType) {
                    $query .= " AND template_type = ?";
                    $params[] = $templateType;
                }
                
                $query .= " ORDER BY is_public DESC, created_at DESC";
                
                $stmt = $db->prepare($query);
                $stmt->execute($params);
                $templates = $stmt->fetchAll();
                
                // Decode JSON data
                foreach ($templates as &$template) {
                    $template['template_data'] = json_decode($template['template_data'], true);
                }
                
                echo json_encode(['success' => true, 'data' => $templates]);
                
            } elseif ($action === 'get') {
                // Get single template
                $templateId = intval($_GET['id'] ?? 0);
                
                $stmt = $db->prepare("SELECT * FROM campaign_templates WHERE id = ? AND (user_id = ? OR is_public = 1)");
                $stmt->execute([$templateId, $userId]);
                $template = $stmt->fetch();
                
                if (!$template) {
                    throw new Exception('Template not found');
                }
                
                $template['template_data'] = json_decode($template['template_data'], true);
                echo json_encode(['success' => true, 'data' => $template]);
            }
            break;
            
        case 'POST':
            requireCSRFToken();
            
            if ($action === 'create') {
                $templateName = trim($_POST['template_name'] ?? '');
                $templateType = $_POST['template_type'] ?? '';
                $templateData = $_POST['template_data'] ?? '{}';
                $description = $_POST['description'] ?? '';
                $isPublic = isset($_POST['is_public']) ? (bool)$_POST['is_public'] : false;
                
                // Validate
                if (empty($templateName) || empty($templateType)) {
                    throw new Exception('Template name and type are required');
                }
                
                // Validate JSON
                $decodedData = json_decode($templateData, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('Invalid template data JSON');
                }
                
                // Only admins can create public templates
                if ($isPublic && !hasRole('admin')) {
                    $isPublic = false;
                }
                
                $stmt = $db->prepare("
                    INSERT INTO campaign_templates 
                    (user_id, template_name, template_type, template_data, description, is_public)
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $userId,
                    $templateName,
                    $templateType,
                    $templateData,
                    $description,
                    $isPublic ? 1 : 0
                ]);
                
                $templateId = $db->lastInsertId();
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Template created successfully',
                    'template_id' => $templateId
                ]);
                
            } elseif ($action === 'use') {
                // Create campaign from template
                $templateId = intval($_POST['template_id'] ?? 0);
                $campaignName = trim($_POST['campaign_name'] ?? '');
                
                if (empty($campaignName)) {
                    throw new Exception('Campaign name is required');
                }
                
                // Get template
                $stmt = $db->prepare("SELECT * FROM campaign_templates WHERE id = ? AND (user_id = ? OR is_public = 1)");
                $stmt->execute([$templateId, $userId]);
                $template = $stmt->fetch();
                
                if (!$template) {
                    throw new Exception('Template not found');
                }
                
                $templateData = json_decode($template['template_data'], true);
                
                // Create campaign from template
                $stmt = $db->prepare("
                    INSERT INTO campaigns 
                    (user_id, campaign_name, campaign_type, budget, start_date, end_date, description, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, 'draft')
                ");
                $stmt->execute([
                    $userId,
                    $campaignName,
                    $template['template_type'],
                    $templateData['budget'] ?? null,
                    $templateData['start_date'] ?? null,
                    $templateData['end_date'] ?? null,
                    $templateData['description'] ?? ''
                ]);
                
                $campaignId = $db->lastInsertId();
                
                // Update template usage count
                $stmt = $db->prepare("UPDATE campaign_templates SET usage_count = usage_count + 1 WHERE id = ?");
                $stmt->execute([$templateId]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Campaign created from template',
                    'campaign_id' => $campaignId
                ]);
            }
            break;
            
        case 'DELETE':
            requireCSRFToken();
            
            if ($action === 'delete') {
                $templateId = intval($_GET['id'] ?? $_POST['id'] ?? 0);
                
                // Check ownership
                $stmt = $db->prepare("SELECT user_id FROM campaign_templates WHERE id = ?");
                $stmt->execute([$templateId]);
                $template = $stmt->fetch();
                
                if (!$template) {
                    throw new Exception('Template not found');
                }
                
                if ($template['user_id'] != $userId && !hasRole('admin')) {
                    throw new Exception('Permission denied');
                }
                
                $stmt = $db->prepare("DELETE FROM campaign_templates WHERE id = ?");
                $stmt->execute([$templateId]);
                
                echo json_encode(['success' => true, 'message' => 'Template deleted']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

