<?php
/**
 * Affiliate Tracking API
 * Handles affiliate click tracking and lead capture
 */

require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/AffiliateTracker.php';
require_once __DIR__ . '/../includes/LeadCapture.php';
require_once __DIR__ . '/../includes/ReferralProcessor.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

try {
    $action = $_GET['action'] ?? $_POST['action'] ?? '';
    
    switch ($action) {
        case 'track_click':
            $partnerCode = trim($_GET['ref'] ?? '');
            $referrerUrl = $_GET['referrer'] ?? null;
            $landingPage = $_GET['landing'] ?? null;
            
            if (empty($partnerCode)) {
                throw new Exception('Partner code is required');
            }
            
            $tracker = new AffiliateTracker();
            $clickId = $tracker->trackClick($partnerCode, $referrerUrl, $landingPage);
            
            if ($clickId) {
                echo json_encode([
                    'success' => true,
                    'click_id' => $clickId,
                    'message' => 'Click tracked successfully'
                ]);
            } else {
                throw new Exception('Failed to track click');
            }
            break;
            
        case 'capture_lead':
            $email = trim($_POST['email'] ?? '');
            $name = trim($_POST['name'] ?? '');
            $firstName = trim($_POST['first_name'] ?? '');
            $lastName = trim($_POST['last_name'] ?? '');
            $phone = trim($_POST['phone'] ?? '');
            $companyName = trim($_POST['company_name'] ?? '');
            $notes = trim($_POST['notes'] ?? '');
            $userId = (int)($_POST['user_id'] ?? 0);
            $campaignId = (int)($_POST['campaign_id'] ?? 0);
            $affiliateCode = trim($_POST['affiliate_code'] ?? '');
            $leadSource = trim($_POST['lead_source'] ?? 'acceleratedleadz');
            
            if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                throw new Exception('Valid email is required');
            }
            
            // Start session to check for affiliate tracking
            if (session_status() === PHP_SESSION_NONE) {
                session_start();
            }
            
            // Get affiliate code from POST or session
            if (empty($affiliateCode) && isset($_SESSION['affiliate_code'])) {
                $affiliateCode = $_SESSION['affiliate_code'];
            }
            
            // If affiliate code provided, track the click first
            if (!empty($affiliateCode)) {
                $tracker = new AffiliateTracker();
                $tracker->trackClick($affiliateCode, $_SERVER['HTTP_REFERER'] ?? null, $_SERVER['REQUEST_URI'] ?? null);
            }
            
            $leadCapture = new LeadCapture();
            $result = $leadCapture->captureLead([
                'email' => $email,
                'name' => $name,
                'phone' => $phone,
                'company_name' => $companyName,
                'notes' => $notes
            ]);
            
            // If result is successful and we have user_id, add to email_subscribers
            if ($result['success'] && $userId) {
                try {
                    $db = getDB();
                    $stmt = $db->prepare("
                        INSERT INTO email_subscribers (user_id, email, first_name, last_name, status, subscribed_at)
                        VALUES (?, ?, ?, ?, 'subscribed', NOW())
                        ON DUPLICATE KEY UPDATE
                            first_name = COALESCE(VALUES(first_name), first_name),
                            last_name = COALESCE(VALUES(last_name), last_name),
                            status = 'subscribed',
                            unsubscribed_at = NULL
                    ");
                    
                    $stmt->execute([
                        $userId,
                        $email,
                        !empty($firstName) ? $firstName : (explode(' ', $name)[0] ?? null),
                        !empty($lastName) ? $lastName : (count(explode(' ', $name)) > 1 ? end(explode(' ', $name)) : null)
                    ]);
                    
                    // Update lead with user_id if lead_id exists
                    if (isset($result['lead_id'])) {
                        $updateStmt = $db->prepare("UPDATE client_leads SET user_id = ? WHERE id = ?");
                        $updateStmt->execute([$userId, $result['lead_id']]);
                    }
                    
                    // Queue email if campaign_id provided
                    if ($campaignId) {
                        $subscriberId = $db->lastInsertId() ?: null;
                        if ($subscriberId) {
                            $queueStmt = $db->prepare("
                                INSERT INTO accelerated_leadz_email_queue 
                                (campaign_id, subscriber_id, email_subject, email_content, email_type, scheduled_at, status)
                                VALUES (?, ?, 'Welcome!', '<h1>Thank you for subscribing!</h1><p>We'll be in touch soon.</p>', 'html', NOW(), 'pending')
                            ");
                            $queueStmt->execute([$campaignId, $subscriberId]);
                        }
                    }
                } catch (PDOException $e) {
                    error_log("Lead capture additional processing error: " . $e->getMessage());
                    // Don't fail the request, just log the error
                }
            }
            
            echo json_encode($result);
            break;
            
        case 'get_affiliate_info':
            $tracker = new AffiliateTracker();
            $affiliate = $tracker->getCurrentAffiliate();
            
            echo json_encode([
                'success' => true,
                'affiliate' => $affiliate
            ]);
            break;
            
        case 'get_stats':
            $partnerId = intval($_GET['partner_id'] ?? 0);
            $dateRange = intval($_GET['days'] ?? 30);
            
            if (!$partnerId) {
                throw new Exception('Partner ID is required');
            }
            
            require_once __DIR__ . '/../includes/PartnerDashboard.php';
            $dashboard = new PartnerDashboard();
            $stats = $dashboard->getDashboardStats($partnerId, $dateRange);
            
            echo json_encode([
                'success' => true,
                'stats' => $stats
            ]);
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
