<?php
/**
 * Ad Builder API
 * Handles saving ads and adding them to campaigns
 */

// Start output buffering to catch any errors/warnings
ob_start();

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/csrf.php';

// Set JSON header early
header('Content-Type: application/json; charset=utf-8');

// Clear any output that might have been generated
ob_clean();

requireAuth();
$db = getDB();
$userId = getEffectiveUserId(); // Use effective user ID (supports admin access)
$userRole = $_SESSION['user_role'];

$method = $_SERVER['REQUEST_METHOD'];
$action = ($method === 'POST') ? ($_POST['action'] ?? '') : ($_GET['action'] ?? '');

// Add table for saved ads if it doesn't exist
try {
    $db->exec("
        CREATE TABLE IF NOT EXISTS saved_ads (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            user_id INT UNSIGNED NOT NULL,
            ad_name VARCHAR(255) NOT NULL,
            ad_data JSON NOT NULL,
            platform ENUM('google_ads', 'facebook_ads') NOT NULL,
            template_type VARCHAR(50),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_id (user_id),
            INDEX idx_platform (platform)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
} catch (PDOException $e) {
    // Table might already exist
}

// Add table for linking ads to campaigns
try {
    $db->exec("
        CREATE TABLE IF NOT EXISTS campaign_ads (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            campaign_id INT UNSIGNED NOT NULL,
            saved_ad_id INT UNSIGNED NULL,
            ad_name VARCHAR(255) NOT NULL,
            ad_platform VARCHAR(50),
            status ENUM('active', 'paused', 'archived') DEFAULT 'active',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (campaign_id) REFERENCES campaigns(id) ON DELETE CASCADE,
            FOREIGN KEY (saved_ad_id) REFERENCES saved_ads(id) ON DELETE SET NULL,
            INDEX idx_campaign_id (campaign_id),
            INDEX idx_saved_ad_id (saved_ad_id),
            INDEX idx_status (status),
            UNIQUE KEY uniq_campaign_ad (campaign_id, saved_ad_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
} catch (PDOException $e) {
    // Table might already exist
}

try {
    if ($method === 'POST') {
        // Validate CSRF token
        $token = $_POST['csrf_token'] ?? '';
        if (!isset($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $token)) {
            ob_clean();
            http_response_code(403);
            echo json_encode([
                'success' => false,
                'error' => 'CSRF token validation failed. Please refresh the page and try again.'
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        }
        
        if ($action === 'save_ad') {
            $adData = json_decode($_POST['ad_data'] ?? '{}', true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid ad data JSON');
            }
            
            // Validate that we have actual user content, not just empty/mock data
            $hasContent = !empty($adData['headline']) || !empty($adData['description']) || !empty($adData['productImage']);
            if (!$hasContent) {
                throw new Exception('Ad must have at least a headline, description, or product image. Please enter ad content before saving.');
            }
            
            $adName = $adData['name'] ?? 'Untitled Ad';
            $platform = $adData['platform'] ?? 'google_ads';
            $templateType = $adData['template'] ?? 'custom';
            
            // Ensure destinationUrl is included in saved data (for tracking/redirect)
            if (!isset($adData['destinationUrl'])) {
                $adData['destinationUrl'] = '';
            }
            
            // Log what we're saving (for debugging, remove in production if needed)
            error_log("Saving ad - Name: {$adName}, Headline: " . ($adData['headline'] ?? 'empty') . ", Destination: " . ($adData['destinationUrl'] ?? 'empty'));
            
            $stmt = $db->prepare("
                INSERT INTO saved_ads (user_id, ad_name, ad_data, platform, template_type)
                VALUES (?, ?, ?, ?, ?)
            ");
            
            // Save the complete ad_data JSON with all user-entered fields
            $stmt->execute([
                $userId,
                $adName,
                json_encode($adData, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
                $platform,
                $templateType
            ]);
            
            $adId = $db->lastInsertId();
            
            ob_clean(); // Clear any output
            echo json_encode([
                'success' => true,
                'message' => 'Ad saved successfully',
                'ad_id' => $adId
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        } elseif ($action === 'add_to_campaign') {
            $campaignId = intval($_POST['campaign_id'] ?? 0);
            $adData = json_decode($_POST['ad_data'] ?? '{}', true);
            
            if (!$campaignId) {
                throw new Exception('Campaign ID is required');
            }
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid ad data JSON');
            }
            
            // Verify campaign ownership
            $stmt = $db->prepare("SELECT user_id, campaign_type, campaign_name FROM campaigns WHERE id = ?");
            $stmt->execute([$campaignId]);
            $campaign = $stmt->fetch();
            
            if (!$campaign || $campaign['user_id'] != $userId) {
                throw new Exception('Campaign not found or access denied');
            }
            
            // Save ad first
            $adName = $adData['name'] ?? 'Campaign Ad';
            $platform = $adData['platform'] ?? 'google_ads';
            $templateType = $adData['template'] ?? 'custom';
            
            // Check if ad already exists in this campaign
            $stmt = $db->prepare("SELECT id FROM saved_ads WHERE user_id = ? AND ad_name = ? LIMIT 1");
            $stmt->execute([$userId, $adName]);
            $existingAd = $stmt->fetch();
            
            if ($existingAd) {
                $savedAdId = $existingAd['id'];
            } else {
                $stmt = $db->prepare("
                    INSERT INTO saved_ads (user_id, ad_name, ad_data, platform, template_type)
                    VALUES (?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $userId,
                    $adName,
                    json_encode($adData),
                    $platform,
                    $templateType
                ]);
                
                $savedAdId = $db->lastInsertId();
            }
            
            // Link ad to campaign
            try {
                $stmt = $db->prepare("
                    INSERT INTO campaign_ads (campaign_id, saved_ad_id, ad_name, ad_platform, status)
                    VALUES (?, ?, ?, ?, 'active')
                    ON DUPLICATE KEY UPDATE 
                        status = 'active',
                        updated_at = NOW()
                ");
                
                $stmt->execute([
                    $campaignId,
                    $savedAdId,
                    $adName,
                    $platform
                ]);
                
                $campaignAdId = $db->lastInsertId() ?: $stmt->rowCount();
            } catch (PDOException $e) {
                // If unique constraint fails, just update the existing record
                $stmt = $db->prepare("
                    UPDATE campaign_ads 
                    SET status = 'active', updated_at = NOW()
                    WHERE campaign_id = ? AND saved_ad_id = ?
                ");
                $stmt->execute([$campaignId, $savedAdId]);
            }
            
            // Create ad performance entry if it's an ad platform campaign
            if (in_array($campaign['campaign_type'], ['google_ads', 'facebook_ads'])) {
                try {
                    $adPlatform = $campaign['campaign_type'];
                    
                    // Check if ad_performance table exists and has the right structure
                    $stmt = $db->prepare("
                        INSERT INTO ad_performance 
                        (user_id, campaign_id, ad_platform, ad_id, ad_name, impressions, clicks, conversions, cost, revenue, date)
                        VALUES (?, ?, ?, ?, ?, 0, 0, 0, 0, 0, CURDATE())
                        ON DUPLICATE KEY UPDATE updated_at = NOW()
                    ");
                    
                    $stmt->execute([
                        $userId,
                        $campaignId,
                        $adPlatform,
                        'ad_' . $savedAdId,
                        $adName
                    ]);
                } catch (PDOException $e) {
                    // Table might not exist or structure is different - log but don't fail
                    error_log("Ad performance entry failed: " . $e->getMessage());
                }
            }
            
            // Ensure proper JSON encoding
            $response = [
                'success' => true,
                'message' => 'Ad added to campaign "' . htmlspecialchars($campaign['campaign_name'], ENT_QUOTES, 'UTF-8') . '" successfully',
                'ad_id' => $savedAdId,
                'campaign_id' => $campaignId,
                'campaign_name' => htmlspecialchars($campaign['campaign_name'], ENT_QUOTES, 'UTF-8')
            ];
            
            ob_clean(); // Clear any output
            echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        } elseif ($action === 'list') {
            $stmt = $db->prepare("
                SELECT * FROM saved_ads 
                WHERE user_id = ? 
                ORDER BY created_at DESC
            ");
            $stmt->execute([$userId]);
            $ads = $stmt->fetchAll();
            
            foreach ($ads as &$ad) {
                $ad['ad_data'] = json_decode($ad['ad_data'], true);
            }
            
            ob_clean();
            echo json_encode(['success' => true, 'data' => $ads], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        } else {
            throw new Exception('Invalid action for POST request');
        }
    } elseif ($method === 'GET') {
        if ($action === 'list') {
            $stmt = $db->prepare("
                SELECT * FROM saved_ads 
                WHERE user_id = ? 
                ORDER BY created_at DESC
            ");
            $stmt->execute([$userId]);
            $ads = $stmt->fetchAll();
            
            foreach ($ads as &$ad) {
                $ad['ad_data'] = json_decode($ad['ad_data'], true);
            }
            
            ob_clean();
            echo json_encode(['success' => true, 'data' => $ads], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        } elseif ($action === 'get') {
            $adId = intval($_GET['id'] ?? 0);
            
            $stmt = $db->prepare("SELECT * FROM saved_ads WHERE id = ? AND user_id = ?");
            $stmt->execute([$adId, $userId]);
            $ad = $stmt->fetch();
            
            if (!$ad) {
                throw new Exception('Ad not found');
            }
            
            $ad['ad_data'] = json_decode($ad['ad_data'], true);
            
            ob_clean();
            echo json_encode(['success' => true, 'data' => $ad], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        } elseif ($action === 'list_campaign_ads') {
            $campaignId = intval($_GET['campaign_id'] ?? 0);
            
            if (!$campaignId) {
                throw new Exception('Campaign ID is required');
            }
            
            // Verify campaign ownership
            $stmt = $db->prepare("SELECT user_id FROM campaigns WHERE id = ?");
            $stmt->execute([$campaignId]);
            $campaign = $stmt->fetch();
            
            if (!$campaign || $campaign['user_id'] != $userId) {
                throw new Exception('Campaign not found or access denied');
            }
            
            // Get ads for this campaign - only return ads that have saved ad data
            $stmt = $db->prepare("
                SELECT ca.id, ca.campaign_id, ca.saved_ad_id, ca.ad_name, ca.ad_platform, ca.status, ca.created_at, ca.updated_at,
                       sa.ad_data, sa.id as saved_ad_id_verify
                FROM campaign_ads ca
                INNER JOIN saved_ads sa ON ca.saved_ad_id = sa.id
                WHERE ca.campaign_id = ? AND ca.status = 'active'
                ORDER BY ca.created_at DESC
            ");
            $stmt->execute([$campaignId]);
            $ads = $stmt->fetchAll();
            
            foreach ($ads as &$ad) {
                // Ensure ad_data is always decoded and available
                if (!empty($ad['ad_data'])) {
                    $decoded = json_decode($ad['ad_data'], true);
                    if ($decoded === null && json_last_error() !== JSON_ERROR_NONE) {
                        error_log("Failed to decode ad_data for campaign_ad_id {$ad['id']}: " . json_last_error_msg());
                        $ad['ad_data'] = [];
                    } else {
                        $ad['ad_data'] = $decoded ?: [];
                    }
                } else {
                    error_log("Warning: Empty ad_data for campaign_ad_id {$ad['id']}, saved_ad_id {$ad['saved_ad_id']}");
                    $ad['ad_data'] = [];
                }
                
                // Validate that we have actual content (not just empty/mock template data)
                $hasRealContent = !empty($ad['ad_data']['headline']) || 
                                  !empty($ad['ad_data']['description']) || 
                                  !empty($ad['ad_data']['productImage']);
                
                if (!$hasRealContent && !empty($ad['ad_data'])) {
                    error_log("Warning: Campaign ad ID {$ad['id']} has ad_data but appears to be empty/mock data. Headline: '" . ($ad['ad_data']['headline'] ?? 'NULL') . "', Description: '" . ($ad['ad_data']['description'] ?? 'NULL') . "'");
                }
                
                // Ensure we have the campaign_ad_id (ca.id)
                if (!isset($ad['campaign_ad_id'])) {
                    $ad['campaign_ad_id'] = $ad['id'];
                }
                
                // Log for debugging - show actual values being returned
                error_log("API returning campaign ad ID {$ad['id']} - Headline: '" . ($ad['ad_data']['headline'] ?? 'EMPTY') . "', Description: '" . substr($ad['ad_data']['description'] ?? 'EMPTY', 0, 50) . "', Has Image: " . (!empty($ad['ad_data']['productImage']) ? 'YES' : 'NO'));
            }
            
            ob_clean();
            echo json_encode(['success' => true, 'data' => $ads], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
        } else {
            throw new Exception('Invalid action for GET request');
        }
    } else {
        throw new Exception('Method not allowed');
    }
} catch (Exception $e) {
    ob_clean(); // Clear any output before sending error
    http_response_code(400);
    echo json_encode([
        'success' => false, 
        'error' => $e->getMessage(),
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
} catch (Throwable $e) {
    ob_clean(); // Clear any output before sending error
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => 'Server error: ' . $e->getMessage(),
        'type' => get_class($e)
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

