<?php
/**
 * AcceleratedLeadz Email Queue Worker
 * Runs every minute via cron to send 1 email per minute (like Mailchimp)
 * 
 * Cron setup (runs every minute):
 * * * * * * /usr/bin/php /path/to/api/acceleratedleadz_worker.php
 */

require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/Mailer.php';
require_once __DIR__ . '/../config/email_config.php';

/**
 * Process queued emails (sends 1 per minute)
 */
function processEmailQueueWorker() {
    $db = getDB();
    
    // Get next pending email that's scheduled to send
    $stmt = $db->prepare("
        SELECT q.*, c.user_id, es.email, es.first_name, es.last_name
        FROM accelerated_leadz_email_queue q
        JOIN accelerated_leadz_campaigns c ON q.campaign_id = c.id
        JOIN email_subscribers es ON q.subscriber_id = es.id
        WHERE q.status = 'pending' 
        AND q.scheduled_at <= NOW()
        AND es.status = 'subscribed'
        ORDER BY q.priority DESC, q.scheduled_at ASC
        LIMIT 1
    ");
    
    $stmt->execute();
    $email = $stmt->fetch();
    
    if (!$email) {
        return ['success' => false, 'message' => 'No emails to send'];
    }
    
    // Mark as sending
    $updateStmt = $db->prepare("
        UPDATE accelerated_leadz_email_queue 
        SET status = 'sending', attempts = attempts + 1
        WHERE id = ?
    ");
    $updateStmt->execute([$email['id']]);
    
    // Send email using Mailer
    try {
        $mailer = new Mailer();
        
        // Personalize email content
        $content = $email['email_content'];
        $content = str_replace('{{first_name}}', $email['first_name'] ?? '', $content);
        $content = str_replace('{{last_name}}', $email['last_name'] ?? '', $content);
        $content = str_replace('{{email}}', $email['email'], $content);
        
        $result = $mailer->send(
            $email['email'],
            $email['email_subject'],
            $content
        );
        
        if ($result['success']) {
            // Mark as sent
            $updateStmt = $db->prepare("
                UPDATE accelerated_leadz_email_queue 
                SET status = 'sent', sent_at = NOW()
                WHERE id = ?
            ");
            $updateStmt->execute([$email['id']]);
            
            // Log send
            $logStmt = $db->prepare("
                INSERT INTO accelerated_leadz_send_logs 
                (queue_id, subscriber_id, campaign_id, email_address, subject, sent_at)
                VALUES (?, ?, ?, ?, ?, NOW())
            ");
            $logStmt->execute([
                $email['id'],
                $email['subscriber_id'],
                $email['campaign_id'],
                $email['email'],
                $email['email_subject']
            ]);
            
            return ['success' => true, 'sent' => $email['id'], 'to' => $email['email']];
        } else {
            // Mark as failed
            $updateStmt = $db->prepare("
                UPDATE accelerated_leadz_email_queue 
                SET status = 'failed', error_message = ?
                WHERE id = ?
            ");
            $updateStmt->execute([$result['error'] ?? 'Unknown error', $email['id']]);
            
            return ['success' => false, 'error' => $result['error'] ?? 'Failed to send'];
        }
    } catch (Exception $e) {
        $updateStmt = $db->prepare("
            UPDATE accelerated_leadz_email_queue 
            SET status = 'failed', error_message = ?
            WHERE id = ?
        ");
        $updateStmt->execute([$e->getMessage(), $email['id']]);
        
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

// Process the email queue
$result = processEmailQueueWorker();

// Log result
if (php_sapi_name() === 'cli') {
    echo json_encode($result, JSON_PRETTY_PRINT) . "\n";
} else {
    header('Content-Type: application/json');
    echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}

