<?php
/**
 * AcceleratedLeadz API
 * AI-IT Inc Marketing Platform
 * 
 * Handles:
 * - IMAP email processing from roundbox
 * - CRM email automation (1 email per minute like Mailchimp)
 * - Drip campaign management
 */

require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/Mailer.php';
require_once __DIR__ . '/../config/email_config.php';

header('Content-Type: application/json; charset=utf-8');
ob_start();

// CORS headers - allow local and production domains
require_once __DIR__ . '/../config/domain_config.php';
$allowedOrigins = [
    'https://ai-itinc.com',
    'http://ai-itinc.local',
    'http://localhost',
    'http://affiliate-it.local'
];
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
if (in_array($origin, $allowedOrigins) || isLocalEnv()) {
    header('Access-Control-Allow-Origin: ' . ($origin ?: '*'));
}
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$action = ($method === 'POST') ? ($_POST['action'] ?? '') : ($_GET['action'] ?? '');

/**
 * Process IMAP emails from roundbox
 */
function processImapEmails($userId = null) {
    if (!function_exists('imap_open')) {
        return ['success' => false, 'error' => 'IMAP extension not enabled'];
    }
    
    $host = IMAP_HOST;
    $port = (int)IMAP_PORT;
    $encryption = IMAP_ENCRYPTION;
    $user = IMAP_USER;
    $pass = IMAP_PASS;
    $mailbox = IMAP_MAILBOX ?: 'INBOX';
    
    // Override with roundbox IMAP settings if available
    if (defined('ROUNDBOX_IMAP_HOST') && !empty(ROUNDBOX_IMAP_HOST)) {
        $host = ROUNDBOX_IMAP_HOST;
        $port = defined('ROUNDBOX_IMAP_PORT') ? (int)ROUNDBOX_IMAP_PORT : 993;
        $user = defined('ROUNDBOX_IMAP_USER') ? ROUNDBOX_IMAP_USER : IMAP_USER;
        $pass = defined('ROUNDBOX_IMAP_PASS') ? ROUNDBOX_IMAP_PASS : IMAP_PASS;
    }
    
    if (!$host || !$user || !$pass) {
        return ['success' => false, 'error' => 'IMAP not configured'];
    }
    
    $flags = ($encryption === 'ssl') ? '/ssl' : (($encryption === 'tls') ? '/tls' : '');
    $mailboxPath = sprintf('{%s:%d%s}%s', $host, $port ?: 993, $flags, $mailbox);
    
    $inbox = @imap_open($mailboxPath, $user, $pass);
    if (!$inbox) {
        return ['success' => false, 'error' => 'Failed to open mailbox: ' . imap_last_error()];
    }
    
    $db = getDB();
    $processed = 0;
    $emails = imap_search($inbox, 'UNSEEN');
    
    if ($emails) {
        rsort($emails);
        foreach ($emails as $emailNumber) {
            $overview = imap_fetch_overview($inbox, $emailNumber, 0);
            $ov = $overview[0] ?? null;
            
            if (!$ov) continue;
            
            // Get email body (prefer HTML, fallback to text)
            $bodyText = imap_fetchbody($inbox, $emailNumber, 1.1);
            $bodyHtml = imap_fetchbody($inbox, $emailNumber, 1.2);
            
            if (empty($bodyHtml)) {
                $bodyHtml = imap_fetchbody($inbox, $emailNumber, 1);
            }
            if (empty($bodyText)) {
                $bodyText = strip_tags($bodyHtml);
            }
            
            $subject = isset($ov->subject) ? imap_utf8($ov->subject) : '';
            $from = $ov->from ?? '';
            $date = $ov->date ?? date('Y-m-d H:i:s');
            $msgId = $ov->message_id ?? null;
            
            // Extract email address from "Name <email@example.com>" format
            $fromEmail = $from;
            if (preg_match('/<(.+)>/', $from, $matches)) {
                $fromEmail = $matches[1];
            }
            $fromName = preg_replace('/<.+>/', '', $from);
            $fromName = trim($fromName, ' "\'');
            
            // Store in accelerated_leadz_imap_emails table
            try {
                $stmt = $db->prepare("
                    INSERT INTO accelerated_leadz_imap_emails 
                    (user_id, message_id, from_email, from_name, subject, body_text, body_html, date_received, imap_source)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'roundbox')
                    ON DUPLICATE KEY UPDATE processed = FALSE
                ");
                
                $stmt->execute([
                    $userId ?: 0,
                    $msgId,
                    $fromEmail,
                    $fromName,
                    $subject,
                    $bodyText,
                    $bodyHtml,
                    date('Y-m-d H:i:s', strtotime($date ?: 'now'))
                ]);
                
                $processed++;
            } catch (PDOException $e) {
                error_log("AcceleratedLeadz IMAP processing error: " . $e->getMessage());
            }
            
            // Mark as seen
            imap_setflag_full($inbox, (string)$emailNumber, "\\Seen");
        }
    }
    
    imap_close($inbox);
    
    return ['success' => true, 'processed' => $processed];
}

/**
 * Queue email to CRM customer (adds to queue for 1-per-minute sending)
 */
function queueEmailToCustomer($campaignId, $subscriberId, $subject, $content, $scheduledAt = null) {
    $db = getDB();
    
    if (!$scheduledAt) {
        // Find next available slot (1 minute after last scheduled email)
        $stmt = $db->prepare("
            SELECT MAX(scheduled_at) as last_scheduled 
            FROM accelerated_leadz_email_queue 
            WHERE campaign_id = ?
        ");
        $stmt->execute([$campaignId]);
        $result = $stmt->fetch();
        $lastScheduled = $result['last_scheduled'] ?? date('Y-m-d H:i:00');
        $scheduledAt = date('Y-m-d H:i:s', strtotime($lastScheduled . ' +1 minute'));
    }
    
    $stmt = $db->prepare("
        INSERT INTO accelerated_leadz_email_queue 
        (campaign_id, subscriber_id, email_subject, email_content, email_type, scheduled_at, status)
        VALUES (?, ?, ?, ?, 'html', ?, 'pending')
    ");
    
    $stmt->execute([$campaignId, $subscriberId, $subject, $content, $scheduledAt]);
    
    return $db->lastInsertId();
}

/**
 * Process queued emails (sends 1 per minute)
 */
function processEmailQueue() {
    $db = getDB();
    
    // Get next pending email that's scheduled to send
    $stmt = $db->prepare("
        SELECT q.*, c.user_id, es.email, es.first_name, es.last_name
        FROM accelerated_leadz_email_queue q
        JOIN accelerated_leadz_campaigns c ON q.campaign_id = c.id
        JOIN email_subscribers es ON q.subscriber_id = es.id
        WHERE q.status = 'pending' 
        AND q.scheduled_at <= NOW()
        AND es.status = 'subscribed'
        ORDER BY q.priority DESC, q.scheduled_at ASC
        LIMIT 1
    ");
    
    $stmt->execute();
    $email = $stmt->fetch();
    
    if (!$email) {
        return ['success' => false, 'message' => 'No emails to send'];
    }
    
    // Mark as sending
    $updateStmt = $db->prepare("
        UPDATE accelerated_leadz_email_queue 
        SET status = 'sending', attempts = attempts + 1
        WHERE id = ?
    ");
    $updateStmt->execute([$email['id']]);
    
    // Send email using Mailer
    try {
        $mailer = new Mailer();
        
        // Personalize email content
        $content = $email['email_content'];
        $content = str_replace('{{first_name}}', $email['first_name'] ?? '', $content);
        $content = str_replace('{{last_name}}', $email['last_name'] ?? '', $content);
        $content = str_replace('{{email}}', $email['email'], $content);
        
        $result = $mailer->send(
            $email['email'],
            $email['email_subject'],
            $content
        );
        
        if ($result['success']) {
            // Mark as sent
            $updateStmt = $db->prepare("
                UPDATE accelerated_leadz_email_queue 
                SET status = 'sent', sent_at = NOW()
                WHERE id = ?
            ");
            $updateStmt->execute([$email['id']]);
            
            // Log send
            $logStmt = $db->prepare("
                INSERT INTO accelerated_leadz_send_logs 
                (queue_id, subscriber_id, campaign_id, email_address, subject, sent_at)
                VALUES (?, ?, ?, ?, ?, NOW())
            ");
            $logStmt->execute([
                $email['id'],
                $email['subscriber_id'],
                $email['campaign_id'],
                $email['email'],
                $email['email_subject']
            ]);
            
            return ['success' => true, 'sent' => $email['id'], 'to' => $email['email']];
        } else {
            // Mark as failed
            $updateStmt = $db->prepare("
                UPDATE accelerated_leadz_email_queue 
                SET status = 'failed', error_message = ?
                WHERE id = ?
            ");
            $updateStmt->execute([$result['error'] ?? 'Unknown error', $email['id']]);
            
            return ['success' => false, 'error' => $result['error'] ?? 'Failed to send'];
        }
    } catch (Exception $e) {
        $updateStmt = $db->prepare("
            UPDATE accelerated_leadz_email_queue 
            SET status = 'failed', error_message = ?
            WHERE id = ?
        ");
        $updateStmt->execute([$e->getMessage(), $email['id']]);
        
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Create automation campaign
 */
function createCampaign($userId, $data) {
    $db = getDB();
    
    $stmt = $db->prepare("
        INSERT INTO accelerated_leadz_campaigns 
        (user_id, campaign_name, campaign_type, status, imap_source, send_rate, start_date, end_date)
        VALUES (?, ?, ?, 'draft', 'roundbox', ?, ?, ?)
    ");
    
    $stmt->execute([
        $userId,
        $data['campaign_name'] ?? 'New Campaign',
        $data['campaign_type'] ?? 'drip',
        $data['send_rate'] ?? 1, // 1 per minute
        $data['start_date'] ?? null,
        $data['end_date'] ?? null
    ]);
    
    return $db->lastInsertId();
}

// Route actions
try {
    ob_clean();
    
    switch ($action) {
        case 'process_imap':
            $userId = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
            $result = processImapEmails($userId);
            echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'queue_email':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            $userId = $_SESSION['user_id'];
            
            $campaignId = (int)($_POST['campaign_id'] ?? 0);
            $subscriberId = (int)($_POST['subscriber_id'] ?? 0);
            $subject = $_POST['subject'] ?? '';
            $content = $_POST['content'] ?? '';
            
            if (!$campaignId || !$subscriberId || !$subject || !$content) {
                echo json_encode(['success' => false, 'error' => 'Missing required fields']);
                exit;
            }
            
            $queueId = queueEmailToCustomer($campaignId, $subscriberId, $subject, $content);
            echo json_encode(['success' => true, 'queue_id' => $queueId], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'process_queue':
            // This can be called by cron every minute
            $result = processEmailQueue();
            echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'create_campaign':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            $userId = $_SESSION['user_id'];
            
            $campaignId = createCampaign($userId, $_POST);
            echo json_encode(['success' => true, 'campaign_id' => $campaignId], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'get_campaigns':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            $userId = $_SESSION['user_id'];
            
            $db = getDB();
            $stmt = $db->prepare("
                SELECT * FROM accelerated_leadz_campaigns 
                WHERE user_id = ? 
                ORDER BY created_at DESC
            ");
            $stmt->execute([$userId]);
            $campaigns = $stmt->fetchAll();
            
            echo json_encode(['success' => true, 'campaigns' => $campaigns], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'get_queue_stats':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            $userId = $_SESSION['user_id'];
            
            $db = getDB();
            $stmt = $db->prepare("
                SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                    SUM(CASE WHEN status = 'sent' THEN 1 ELSE 0 END) as sent,
                    SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
                FROM accelerated_leadz_email_queue q
                JOIN accelerated_leadz_campaigns c ON q.campaign_id = c.id
                WHERE c.user_id = ?
            ");
            $stmt->execute([$userId]);
            $stats = $stmt->fetch();
            
            echo json_encode(['success' => true, 'stats' => $stats], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Invalid action'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
    }
} catch (Exception $e) {
    ob_clean();
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

