<?php
/**
 * AcceleratedAdz API
 * AI-IT Inc Native Ad Platform
 * Similar to Facebook Ads and Google Ads, hosted on ai-itinc.com domain
 */

require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/functions.php';

header('Content-Type: application/json; charset=utf-8');
ob_start();

// CORS headers - allow local and production domains
require_once __DIR__ . '/../config/domain_config.php';
$allowedOrigins = [
    'https://ai-itinc.com',
    'http://ai-itinc.local',
    'http://localhost',
    'http://affiliate-it.local'
];
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
if (in_array($origin, $allowedOrigins) || isLocalEnv()) {
    header('Access-Control-Allow-Origin: ' . ($origin ?: '*'));
}
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$action = ($method === 'POST') ? ($_POST['action'] ?? '') : ($_GET['action'] ?? '');

/**
 * Create AcceleratedAdz campaign
 */
function createCampaign($userId, $data) {
    $db = getDB();
    
    $stmt = $db->prepare("
        INSERT INTO acceleratedadz_campaigns 
        (user_id, campaign_name, campaign_type, status, objective, budget_type, daily_budget, lifetime_budget, 
         start_date, end_date, targeting, optimization_goal, bid_strategy, default_bid, domain)
        VALUES (?, ?, ?, 'draft', ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'ai-itinc.com')
    ");
    
    $targeting = json_encode($data['targeting'] ?? []);
    
    $stmt->execute([
        $userId,
        $data['campaign_name'] ?? 'New Campaign',
        $data['campaign_type'] ?? 'display',
        $data['objective'] ?? 'traffic',
        $data['budget_type'] ?? 'daily',
        $data['daily_budget'] ?? null,
        $data['lifetime_budget'] ?? null,
        $data['start_date'] ?? null,
        $data['end_date'] ?? null,
        $targeting,
        $data['optimization_goal'] ?? 'clicks',
        $data['bid_strategy'] ?? 'cpc',
        $data['default_bid'] ?? 1.00
    ]);
    
    return $db->lastInsertId();
}

/**
 * Create ad group
 */
function createAdGroup($campaignId, $data) {
    $db = getDB();
    
    // Get campaign_id for denormalization
    $campStmt = $db->prepare("SELECT id FROM acceleratedadz_campaigns WHERE id = ?");
    $campStmt->execute([$campaignId]);
    $campaign = $campStmt->fetch();
    
    if (!$campaign) {
        return ['success' => false, 'error' => 'Campaign not found'];
    }
    
    $stmt = $db->prepare("
        INSERT INTO acceleratedadz_ad_groups 
        (campaign_id, ad_group_name, status, daily_budget, bid_amount, targeting)
        VALUES (?, ?, 'active', ?, ?, ?)
    ");
    
    $targeting = json_encode($data['targeting'] ?? []);
    
    $stmt->execute([
        $campaignId,
        $data['ad_group_name'] ?? 'New Ad Group',
        $data['daily_budget'] ?? null,
        $data['bid_amount'] ?? null,
        $targeting
    ]);
    
    return $db->lastInsertId();
}

/**
 * Create ad
 */
function createAd($adGroupId, $campaignId, $data) {
    $db = getDB();
    
    $stmt = $db->prepare("
        INSERT INTO acceleratedadz_ads 
        (ad_group_id, campaign_id, ad_name, ad_type, status, headline, description, 
         display_url, destination_url, image_url, video_url, ad_data, call_to_action)
        VALUES (?, ?, ?, ?, 'pending_review', ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $adData = json_encode($data['ad_data'] ?? []);
    
    $stmt->execute([
        $adGroupId,
        $campaignId,
        $data['ad_name'] ?? 'New Ad',
        $data['ad_type'] ?? 'image',
        $data['headline'] ?? '',
        $data['description'] ?? '',
        $data['display_url'] ?? '',
        $data['destination_url'] ?? '',
        $data['image_url'] ?? '',
        $data['video_url'] ?? '',
        $adData,
        $data['call_to_action'] ?? 'Learn More'
    ]);
    
    return $db->lastInsertId();
}

/**
 * Update ad performance (called by tracking system)
 */
function updateAdPerformance($adId, $date, $metrics) {
    $db = getDB();
    
    // Get ad group and campaign IDs
    $adStmt = $db->prepare("SELECT ad_group_id, campaign_id FROM acceleratedadz_ads WHERE id = ?");
    $adStmt->execute([$adId]);
    $ad = $adStmt->fetch();
    
    if (!$ad) {
        return ['success' => false, 'error' => 'Ad not found'];
    }
    
    // Calculate metrics
    $impressions = (int)($metrics['impressions'] ?? 0);
    $clicks = (int)($metrics['clicks'] ?? 0);
    $conversions = (int)($metrics['conversions'] ?? 0);
    $spend = (float)($metrics['spend'] ?? 0.00);
    $revenue = (float)($metrics['revenue'] ?? 0.00);
    
    $ctr = $impressions > 0 ? ($clicks / $impressions) : 0;
    $cpc = $clicks > 0 ? ($spend / $clicks) : 0;
    $cpm = $impressions > 0 ? (($spend / $impressions) * 1000) : 0;
    $cpa = $conversions > 0 ? ($spend / $conversions) : 0;
    $roas = $spend > 0 ? ($revenue / $spend) : 0;
    
    $stmt = $db->prepare("
        INSERT INTO acceleratedadz_performance 
        (ad_id, ad_group_id, campaign_id, date, impressions, clicks, conversions, spend, revenue, 
         ctr, cpc, cpm, cpa, roas)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
            impressions = impressions + VALUES(impressions),
            clicks = clicks + VALUES(clicks),
            conversions = conversions + VALUES(conversions),
            spend = spend + VALUES(spend),
            revenue = revenue + VALUES(revenue),
            ctr = CASE WHEN (impressions + VALUES(impressions)) > 0 
                THEN (clicks + VALUES(clicks)) / (impressions + VALUES(impressions)) 
                ELSE 0 END,
            cpc = CASE WHEN (clicks + VALUES(clicks)) > 0 
                THEN (spend + VALUES(spend)) / (clicks + VALUES(clicks)) 
                ELSE 0 END,
            cpm = CASE WHEN (impressions + VALUES(impressions)) > 0 
                THEN ((spend + VALUES(spend)) / (impressions + VALUES(impressions))) * 1000 
                ELSE 0 END,
            cpa = CASE WHEN (conversions + VALUES(conversions)) > 0 
                THEN (spend + VALUES(spend)) / (conversions + VALUES(conversions)) 
                ELSE 0 END,
            roas = CASE WHEN (spend + VALUES(spend)) > 0 
                THEN (revenue + VALUES(revenue)) / (spend + VALUES(spend)) 
                ELSE 0 END,
            updated_at = NOW()
    ");
    
    $stmt->execute([
        $adId,
        $ad['ad_group_id'],
        $ad['campaign_id'],
        $date,
        $impressions,
        $clicks,
        $conversions,
        $spend,
        $revenue,
        $ctr,
        $cpc,
        $cpm,
        $cpa,
        $roas
    ]);
    
    return ['success' => true];
}

/**
 * Get campaign performance
 */
function getCampaignPerformance($campaignId, $startDate = null, $endDate = null) {
    $db = getDB();
    
    $where = ['campaign_id = ?'];
    $params = [$campaignId];
    
    if ($startDate) {
        $where[] = 'date >= ?';
        $params[] = $startDate;
    }
    
    if ($endDate) {
        $where[] = 'date <= ?';
        $params[] = $endDate;
    }
    
    $sql = "
        SELECT 
            date,
            SUM(impressions) as impressions,
            SUM(clicks) as clicks,
            SUM(conversions) as conversions,
            SUM(spend) as spend,
            SUM(revenue) as revenue,
            CASE WHEN SUM(impressions) > 0 
                THEN SUM(clicks) / SUM(impressions) 
                ELSE 0 END as ctr,
            CASE WHEN SUM(clicks) > 0 
                THEN SUM(spend) / SUM(clicks) 
                ELSE 0 END as cpc,
            CASE WHEN SUM(impressions) > 0 
                THEN (SUM(spend) / SUM(impressions)) * 1000 
                ELSE 0 END as cpm,
            CASE WHEN SUM(conversions) > 0 
                THEN SUM(spend) / SUM(conversions) 
                ELSE 0 END as cpa,
            CASE WHEN SUM(spend) > 0 
                THEN SUM(revenue) / SUM(spend) 
                ELSE 0 END as roas
        FROM acceleratedadz_performance
        WHERE " . implode(' AND ', $where) . "
        GROUP BY date
        ORDER BY date DESC
    ";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    
    return $stmt->fetchAll();
}

// Route actions
try {
    ob_clean();
    
    switch ($action) {
        case 'create_campaign':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            $userId = $_SESSION['user_id'];
            
            $campaignId = createCampaign($userId, $_POST);
            echo json_encode(['success' => true, 'campaign_id' => $campaignId], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'get_campaigns':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            $userId = $_SESSION['user_id'];
            
            $db = getDB();
            $stmt = $db->prepare("
                SELECT c.*, 
                    COUNT(DISTINCT ag.id) as ad_group_count,
                    COUNT(DISTINCT a.id) as ad_count,
                    COALESCE(SUM(p.impressions), 0) as total_impressions,
                    COALESCE(SUM(p.clicks), 0) as total_clicks,
                    COALESCE(SUM(p.spend), 0) as total_spend,
                    COALESCE(SUM(p.revenue), 0) as total_revenue
                FROM acceleratedadz_campaigns c
                LEFT JOIN acceleratedadz_ad_groups ag ON c.id = ag.campaign_id
                LEFT JOIN acceleratedadz_ads a ON ag.id = a.ad_group_id
                LEFT JOIN acceleratedadz_performance p ON a.id = p.ad_id
                WHERE c.user_id = ?
                GROUP BY c.id
                ORDER BY c.created_at DESC
            ");
            $stmt->execute([$userId]);
            $campaigns = $stmt->fetchAll();
            
            // Decode JSON fields
            foreach ($campaigns as &$camp) {
                $camp['targeting'] = json_decode($camp['targeting'] ?? '{}', true);
            }
            
            echo json_encode(['success' => true, 'campaigns' => $campaigns], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'create_ad_group':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            
            $campaignId = (int)($_POST['campaign_id'] ?? 0);
            if (!$campaignId) {
                echo json_encode(['success' => false, 'error' => 'Campaign ID required']);
                exit;
            }
            
            $adGroupId = createAdGroup($campaignId, $_POST);
            echo json_encode(['success' => true, 'ad_group_id' => $adGroupId], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'create_ad':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            
            $adGroupId = (int)($_POST['ad_group_id'] ?? 0);
            $campaignId = (int)($_POST['campaign_id'] ?? 0);
            
            if (!$adGroupId || !$campaignId) {
                echo json_encode(['success' => false, 'error' => 'Ad Group ID and Campaign ID required']);
                exit;
            }
            
            $adId = createAd($adGroupId, $campaignId, $_POST);
            echo json_encode(['success' => true, 'ad_id' => $adId], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'track_click':
            // Track ad click (public, no auth required)
            $adId = (int)($_POST['ad_id'] ?? $_GET['ad_id'] ?? 0);
            $date = date('Y-m-d');
            
            if (!$adId) {
                echo json_encode(['success' => false, 'error' => 'Ad ID required']);
                exit;
            }
            
            // Get ad info
            $db = getDB();
            $adStmt = $db->prepare("SELECT ad_group_id, campaign_id FROM acceleratedadz_ads WHERE id = ?");
            $adStmt->execute([$adId]);
            $ad = $adStmt->fetch();
            
            if (!$ad) {
                echo json_encode(['success' => false, 'error' => 'Ad not found']);
                exit;
            }
            
            // Increment clicks
            $stmt = $db->prepare("
                INSERT INTO acceleratedadz_performance 
                (ad_id, ad_group_id, campaign_id, date, impressions, clicks, conversions, spend, revenue, 
                 ctr, cpc, cpm, cpa, roas)
                VALUES (?, ?, ?, ?, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0)
                ON DUPLICATE KEY UPDATE
                    clicks = clicks + 1,
                    ctr = CASE WHEN impressions > 0 THEN (clicks + 1) / impressions ELSE 0 END,
                    updated_at = NOW()
            ");
            $stmt->execute([$adId, $ad['ad_group_id'], $ad['campaign_id'], $date]);
            
            echo json_encode(['success' => true, 'clicked' => $adId], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'update_performance':
            // Can be called by tracking system without auth
            $adId = (int)($_POST['ad_id'] ?? 0);
            $date = $_POST['date'] ?? date('Y-m-d');
            $metrics = $_POST['metrics'] ?? [];
            
            if (!$adId) {
                echo json_encode(['success' => false, 'error' => 'Ad ID required']);
                exit;
            }
            
            $result = updateAdPerformance($adId, $date, $metrics);
            echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        case 'get_performance':
            require_once __DIR__ . '/../includes/auth.php';
            requireClient();
            
            $campaignId = (int)($_GET['campaign_id'] ?? 0);
            $startDate = $_GET['start_date'] ?? null;
            $endDate = $_GET['end_date'] ?? null;
            
            if (!$campaignId) {
                echo json_encode(['success' => false, 'error' => 'Campaign ID required']);
                exit;
            }
            
            $performance = getCampaignPerformance($campaignId, $startDate, $endDate);
            echo json_encode(['success' => true, 'performance' => $performance], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Invalid action'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            exit;
    }
} catch (Exception $e) {
    ob_clean();
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

