<?php
/**
 * A/B Testing API
 * Handles campaign variant creation and management
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');
requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if ($action === 'variants') {
                $campaignId = intval($_GET['campaign_id'] ?? 0);
                
                if (!$campaignId) {
                    throw new Exception('Campaign ID is required');
                }
                
                // Verify campaign ownership
                $stmt = $db->prepare("SELECT user_id FROM campaigns WHERE id = ?");
                $stmt->execute([$campaignId]);
                $campaign = $stmt->fetch();
                
                if (!$campaign || $campaign['user_id'] != $userId) {
                    throw new Exception('Campaign not found or access denied');
                }
                
                $stmt = $db->prepare("SELECT * FROM campaign_variants WHERE campaign_id = ? ORDER BY created_at ASC");
                $stmt->execute([$campaignId]);
                $variants = $stmt->fetchAll();
                
                foreach ($variants as &$variant) {
                    $variant['variant_data'] = json_decode($variant['variant_data'], true);
                }
                
                echo json_encode(['success' => true, 'data' => $variants]);
            }
            break;
            
        case 'POST':
            requireCSRFToken();
            
            if ($action === 'create_variant') {
                $campaignId = intval($_POST['campaign_id'] ?? 0);
                $variantName = trim($_POST['variant_name'] ?? '');
                $variantData = $_POST['variant_data'] ?? '{}';
                $trafficPercentage = floatval($_POST['traffic_percentage'] ?? 50.0);
                
                if (!$campaignId || empty($variantName)) {
                    throw new Exception('Campaign ID and variant name are required');
                }
                
                // Verify campaign ownership
                $stmt = $db->prepare("SELECT user_id FROM campaigns WHERE id = ?");
                $stmt->execute([$campaignId]);
                $campaign = $stmt->fetch();
                
                if (!$campaign || $campaign['user_id'] != $userId) {
                    throw new Exception('Campaign not found or access denied');
                }
                
                // Validate traffic percentage
                if ($trafficPercentage < 0 || $trafficPercentage > 100) {
                    throw new Exception('Traffic percentage must be between 0 and 100');
                }
                
                // Check total traffic allocation
                $stmt = $db->prepare("SELECT SUM(traffic_percentage) as total FROM campaign_variants WHERE campaign_id = ?");
                $stmt->execute([$campaignId]);
                $total = $stmt->fetchColumn();
                
                if (($total + $trafficPercentage) > 100) {
                    throw new Exception('Total traffic allocation cannot exceed 100%');
                }
                
                // Validate JSON
                $decodedData = json_decode($variantData, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('Invalid variant data JSON');
                }
                
                $stmt = $db->prepare("
                    INSERT INTO campaign_variants 
                    (campaign_id, variant_name, variant_data, traffic_percentage)
                    VALUES (?, ?, ?, ?)
                ");
                $stmt->execute([$campaignId, $variantName, $variantData, $trafficPercentage]);
                
                $variantId = $db->lastInsertId();
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Variant created successfully',
                    'variant_id' => $variantId
                ]);
                
            } elseif ($action === 'update_performance') {
                // Update variant performance metrics
                $variantId = intval($_POST['variant_id'] ?? 0);
                $impressions = intval($_POST['impressions'] ?? 0);
                $clicks = intval($_POST['clicks'] ?? 0);
                $conversions = intval($_POST['conversions'] ?? 0);
                $cost = floatval($_POST['cost'] ?? 0);
                $revenue = floatval($_POST['revenue'] ?? 0);
                
                // Verify variant ownership
                $stmt = $db->prepare("
                    SELECT cv.id, c.user_id 
                    FROM campaign_variants cv
                    JOIN campaigns c ON cv.campaign_id = c.id
                    WHERE cv.id = ?
                ");
                $stmt->execute([$variantId]);
                $variant = $stmt->fetch();
                
                if (!$variant || $variant['user_id'] != $userId) {
                    throw new Exception('Variant not found or access denied');
                }
                
                $stmt = $db->prepare("
                    UPDATE campaign_variants 
                    SET impressions = impressions + ?, 
                        clicks = clicks + ?,
                        conversions = conversions + ?,
                        cost = cost + ?,
                        revenue = revenue + ?,
                        updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$impressions, $clicks, $conversions, $cost, $revenue, $variantId]);
                
                echo json_encode(['success' => true, 'message' => 'Performance updated']);
                
            } elseif ($action === 'set_winner') {
                // Mark winning variant (automatically pauses others)
                $variantId = intval($_POST['variant_id'] ?? 0);
                
                // Verify variant ownership
                $stmt = $db->prepare("
                    SELECT cv.id, cv.campaign_id, c.user_id 
                    FROM campaign_variants cv
                    JOIN campaigns c ON cv.campaign_id = c.id
                    WHERE cv.id = ?
                ");
                $stmt->execute([$variantId]);
                $variant = $stmt->fetch();
                
                if (!$variant || $variant['user_id'] != $userId) {
                    throw new Exception('Variant not found or access denied');
                }
                
                $db->beginTransaction();
                try {
                    // Set all variants to not winner
                    $stmt = $db->prepare("UPDATE campaign_variants SET is_winner = 0 WHERE campaign_id = ?");
                    $stmt->execute([$variant['campaign_id']]);
                    
                    // Set this variant as winner and activate it
                    $stmt = $db->prepare("UPDATE campaign_variants SET is_winner = 1, status = 'active' WHERE id = ?");
                    $stmt->execute([$variantId]);
                    
                    // Pause other variants
                    $stmt = $db->prepare("UPDATE campaign_variants SET status = 'paused' WHERE campaign_id = ? AND id != ?");
                    $stmt->execute([$variant['campaign_id'], $variantId]);
                    
                    $db->commit();
                    
                    echo json_encode(['success' => true, 'message' => 'Winner set successfully']);
                } catch (Exception $e) {
                    $db->rollBack();
                    throw $e;
                }
                
            } elseif ($action === 'update_status') {
                // Update variant status (pause/resume)
                $variantId = intval($_POST['variant_id'] ?? 0);
                $status = $_POST['status'] ?? 'paused';
                
                if (!in_array($status, ['active', 'paused', 'completed'])) {
                    throw new Exception('Invalid status');
                }
                
                // Verify variant ownership
                $stmt = $db->prepare("
                    SELECT cv.id, c.user_id 
                    FROM campaign_variants cv
                    JOIN campaigns c ON cv.campaign_id = c.id
                    WHERE cv.id = ?
                ");
                $stmt->execute([$variantId]);
                $variant = $stmt->fetch();
                
                if (!$variant || $variant['user_id'] != $userId) {
                    throw new Exception('Variant not found or access denied');
                }
                
                $stmt = $db->prepare("UPDATE campaign_variants SET status = ? WHERE id = ?");
                $stmt->execute([$status, $variantId]);
                
                echo json_encode(['success' => true, 'message' => "Variant status updated to {$status}"]);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

