<?php
/**
 * Admin - Manage Individual Subscription
 * Handles creating, editing, upgrading/downgrading subscriptions
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireAdmin();
$pageTitle = 'Manage Subscription';

$db = getDB();
$clientId = isset($_GET['client_id']) ? intval($_GET['client_id']) : 0;

if (!$clientId) {
    setFlashMessage('danger', 'Client ID is required');
    header('Location: ' . url('admin/clients.php'));
    exit;
}

// Get client info
$stmt = $db->prepare("SELECT * FROM users WHERE id = ? AND role = 'client'");
$stmt->execute([$clientId]);
$client = $stmt->fetch();

if (!$client) {
    setFlashMessage('danger', 'Client not found');
    header('Location: ' . url('admin/clients.php'));
    exit;
}

// Get current subscription
$stmt = $db->prepare("
    SELECT s.*, st.tier_name, st.tier_level, st.monthly_price
    FROM subscriptions s
    JOIN subscription_tiers st ON s.tier_id = st.id
    WHERE s.user_id = ? AND s.status = 'active'
    ORDER BY s.created_at DESC
    LIMIT 1
");
$stmt->execute([$clientId]);
$currentSubscription = $stmt->fetch();

// Get all tiers
$stmt = $db->query("SELECT * FROM subscription_tiers ORDER BY tier_level");
$tiers = $stmt->fetchAll();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create_subscription') {
        $tierId = intval($_POST['tier_id'] ?? 0);
        $startDate = $_POST['start_date'] ?? date('Y-m-d');
        $renewalDate = $_POST['renewal_date'] ?? date('Y-m-d', strtotime('+1 month'));
        
        // Cancel existing active subscription
        if ($currentSubscription) {
            $stmt = $db->prepare("UPDATE subscriptions SET status = 'cancelled' WHERE id = ?");
            $stmt->execute([$currentSubscription['id']]);
        }
        
        // Create new subscription
        $stmt = $db->prepare("
            INSERT INTO subscriptions (user_id, tier_id, start_date, renewal_date, status, auto_renew, payment_status)
            VALUES (?, ?, ?, ?, 'active', 1, 'paid')
        ");
        $stmt->execute([$clientId, $tierId, $startDate, $renewalDate]);
        
        // Create transaction record
        $subscriptionId = $db->lastInsertId();
        $tierPrice = 0;
        foreach ($tiers as $tier) {
            if ($tier['id'] == $tierId) {
                $tierPrice = $tier['monthly_price'];
                break;
            }
        }
        
        $stmt = $db->prepare("
            INSERT INTO transactions (user_id, subscription_id, transaction_type, amount, payment_status, description)
            VALUES (?, ?, 'subscription', ?, 'completed', 'Subscription created by admin')
        ");
        $stmt->execute([$clientId, $subscriptionId, $tierPrice]);
        
        setFlashMessage('success', 'Subscription created successfully!');
        header('Location: ' . url('admin/subscription_manage.php?client_id=' . $clientId));
        exit;
        
    } elseif ($action === 'update_subscription') {
        $subscriptionId = intval($_POST['subscription_id'] ?? 0);
        $tierId = intval($_POST['tier_id'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        $renewalDate = $_POST['renewal_date'] ?? null;
        $autoRenew = isset($_POST['auto_renew']) ? 1 : 0;
        
        $stmt = $db->prepare("
            UPDATE subscriptions 
            SET tier_id = ?, status = ?, renewal_date = ?, auto_renew = ?, updated_at = NOW()
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$tierId, $status, $renewalDate, $autoRenew, $subscriptionId, $clientId]);
        
        setFlashMessage('success', 'Subscription updated successfully!');
        header('Location: ' . url('admin/subscription_manage.php?client_id=' . $clientId));
        exit;
        
    } elseif ($action === 'cancel_subscription') {
        $subscriptionId = intval($_POST['subscription_id'] ?? 0);
        
        $stmt = $db->prepare("UPDATE subscriptions SET status = 'cancelled', auto_renew = 0 WHERE id = ? AND user_id = ?");
        $stmt->execute([$subscriptionId, $clientId]);
        
        setFlashMessage('success', 'Subscription cancelled successfully!');
        header('Location: ' . url('admin/subscription_manage.php?client_id=' . $clientId));
        exit;
    }
}

// Get transaction history
$stmt = $db->prepare("
    SELECT * FROM transactions 
    WHERE user_id = ? 
    ORDER BY created_at DESC 
    LIMIT 10
");
$stmt->execute([$clientId]);
$transactions = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <h1>Manage Subscription: <?= escape($client['business_name']) ?></h1>
        <p class="text-muted">Email: <?= escape($client['email']) ?></p>
    </div>
</div>

<div class="row">
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <?php if ($currentSubscription): ?>
                        Current Subscription
                    <?php else: ?>
                        No Active Subscription
                    <?php endif; ?>
                </h5>
            </div>
            <div class="card-body">
                <?php if ($currentSubscription): ?>
                    <form method="POST" action="<?= url('admin/subscription_manage.php?client_id=' . $clientId) ?>">
                        <input type="hidden" name="action" value="update_subscription">
                        <input type="hidden" name="subscription_id" value="<?= $currentSubscription['id'] ?>">
                        
                        <div class="mb-3">
                            <label class="form-label">Tier</label>
                            <select class="form-select" name="tier_id" required>
                                <?php foreach ($tiers as $tier): ?>
                                    <option value="<?= $tier['id'] ?>" <?= $tier['id'] == $currentSubscription['tier_id'] ? 'selected' : '' ?>>
                                        <?= escape($tier['tier_name']) ?> - <?= formatCurrency($tier['monthly_price']) ?>/month
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Status</label>
                            <select class="form-select" name="status" required>
                                <option value="active" <?= $currentSubscription['status'] === 'active' ? 'selected' : '' ?>>Active</option>
                                <option value="cancelled" <?= $currentSubscription['status'] === 'cancelled' ? 'selected' : '' ?>>Cancelled</option>
                                <option value="expired" <?= $currentSubscription['status'] === 'expired' ? 'selected' : '' ?>>Expired</option>
                                <option value="pending" <?= $currentSubscription['status'] === 'pending' ? 'selected' : '' ?>>Pending</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Renewal Date</label>
                            <input type="date" class="form-control" name="renewal_date" value="<?= $currentSubscription['renewal_date'] ?>" required>
                        </div>
                        
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="auto_renew" id="auto_renew" <?= $currentSubscription['auto_renew'] ? 'checked' : '' ?>>
                                <label class="form-check-label" for="auto_renew">
                                    Auto Renew
                                </label>
                            </div>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">Update Subscription</button>
                        </div>
                    </form>
                    
                    <hr>
                    
                    <form method="POST" action="/admin/subscription_manage.php?client_id=<?= $clientId ?>" onsubmit="return confirm('Are you sure you want to cancel this subscription?');">
                        <input type="hidden" name="action" value="cancel_subscription">
                        <input type="hidden" name="subscription_id" value="<?= $currentSubscription['id'] ?>">
                        <button type="submit" class="btn btn-danger w-100">Cancel Subscription</button>
                    </form>
                <?php else: ?>
                    <p class="text-muted">This client does not have an active subscription.</p>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createSubscriptionModal">
                        Create Subscription
                    </button>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Transaction History</h5>
            </div>
            <div class="card-body">
                <?php if (empty($transactions)): ?>
                    <p class="text-muted mb-0">No transactions yet.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Type</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transactions as $transaction): ?>
                                    <tr>
                                        <td><?= formatDate($transaction['created_at'], 'Y-m-d') ?></td>
                                        <td><?= ucfirst($transaction['transaction_type']) ?></td>
                                        <td><?= formatCurrency($transaction['amount']) ?></td>
                                        <td>
                                            <span class="badge bg-<?= $transaction['payment_status'] === 'completed' ? 'success' : 'warning' ?>">
                                                <?= ucfirst($transaction['payment_status']) ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Create Subscription Modal -->
<div class="modal fade" id="createSubscriptionModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create New Subscription</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="/admin/subscription_manage.php?client_id=<?= $clientId ?>">
                <div class="modal-body">
                    <input type="hidden" name="action" value="create_subscription">
                    
                    <div class="mb-3">
                        <label class="form-label">Subscription Tier</label>
                        <select class="form-select" name="tier_id" required>
                            <option value="">Select tier...</option>
                            <?php foreach ($tiers as $tier): ?>
                                <option value="<?= $tier['id'] ?>">
                                    <?= escape($tier['tier_name']) ?> - <?= formatCurrency($tier['monthly_price']) ?>/month
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Start Date</label>
                        <input type="date" class="form-control" name="start_date" value="<?= date('Y-m-d') ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Renewal Date</label>
                        <input type="date" class="form-control" name="renewal_date" value="<?= date('Y-m-d', strtotime('+1 month')) ?>" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Subscription</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

