<?php
/**
 * Admin - Affiliate Payout Management
 * Process and manage affiliate payouts
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/csrf.php';

requireAdmin();
$pageTitle = 'Affiliate Payouts';

$db = getDB();
$error = '';
$success = '';

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Search and Filter
$search = trim($_GET['search'] ?? '');
$statusFilter = $_GET['status'] ?? 'all';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requireCSRFToken();
    
    $action = $_POST['action'] ?? '';
    
    if ($action === 'process_payout') {
        $partnerId = intval($_POST['partner_id'] ?? 0);
        $payoutPeriodStart = $_POST['payout_period_start'] ?? '';
        $payoutPeriodEnd = $_POST['payout_period_end'] ?? '';
        $paymentMethod = $_POST['payment_method'] ?? '';
        $paymentReference = trim($_POST['payment_reference'] ?? '');
        
        if (!$partnerId || !$payoutPeriodStart || !$payoutPeriodEnd || !$paymentMethod) {
            $error = 'All fields are required';
        } else {
            try {
                // Get pending conversions for this partner in the period
                $stmt = $db->prepare("
                    SELECT SUM(commission_amount) as total_commission, COUNT(*) as conversion_count
                    FROM affiliate_conversions
                    WHERE partner_id = ? AND commission_status = 'pending' 
                    AND converted_at >= ? AND converted_at <= ?
                ");
                $stmt->execute([$partnerId, $payoutPeriodStart, $payoutPeriodEnd]);
                $totals = $stmt->fetch();
                
                if ($totals['total_commission'] > 0) {
                    // Create payout record
                    $stmt = $db->prepare("
                        INSERT INTO affiliate_payouts 
                        (partner_id, payout_period_start, payout_period_end, total_commission, 
                         conversion_count, payment_method, payment_reference, processed_by)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([
                        $partnerId, $payoutPeriodStart, $payoutPeriodEnd, 
                        $totals['total_commission'], $totals['conversion_count'],
                        $paymentMethod, $paymentReference, $_SESSION['user_id']
                    ]);
                    
                    $payoutId = $db->lastInsertId();
                    
                    // Link conversions to payout
                    $stmt = $db->prepare("
                        INSERT INTO payout_conversions (payout_id, conversion_id)
                        SELECT ?, id FROM affiliate_conversions
                        WHERE partner_id = ? AND commission_status = 'pending' 
                        AND converted_at >= ? AND converted_at <= ?
                    ");
                    $stmt->execute([$payoutId, $partnerId, $payoutPeriodStart, $payoutPeriodEnd]);
                    
                    // Update conversion status
                    $stmt = $db->prepare("
                        UPDATE affiliate_conversions 
                        SET commission_status = 'approved', approved_at = NOW()
                        WHERE partner_id = ? AND commission_status = 'pending' 
                        AND converted_at >= ? AND converted_at <= ?
                    ");
                    $stmt->execute([$partnerId, $payoutPeriodStart, $payoutPeriodEnd]);
                    
                    $success = "Payout processed successfully! Total: $" . number_format($totals['total_commission'], 2);
                } else {
                    $error = 'No pending commissions found for this period';
                }
            } catch (PDOException $e) {
                $error = 'Error processing payout: ' . $e->getMessage();
            }
        }
    } elseif ($action === 'mark_paid') {
        $payoutId = intval($_POST['payout_id'] ?? 0);
        $paymentDate = $_POST['payment_date'] ?? date('Y-m-d');
        
        if ($payoutId) {
            $stmt = $db->prepare("
                UPDATE affiliate_payouts 
                SET payout_status = 'paid', payment_date = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$paymentDate, $payoutId]);
            
            // Update conversion status to paid
            $stmt = $db->prepare("
                UPDATE affiliate_conversions ac
                JOIN payout_conversions pc ON ac.id = pc.conversion_id
                SET ac.commission_status = 'paid'
                WHERE pc.payout_id = ?
            ");
            $stmt->execute([$payoutId]);
            
            $success = 'Payout marked as paid';
        }
    }
}

// Build query with filters
$whereClauses = [];
$params = [];

if (!empty($search)) {
    $whereClauses[] = "(ap.partner_name LIKE ? OR ap.contact_email LIKE ?)";
    $searchTerm = '%' . $search . '%';
    $params = array_merge($params, [$searchTerm, $searchTerm]);
}

if ($statusFilter !== 'all') {
    $whereClauses[] = "payout_status = ?";
    $params[] = $statusFilter;
}

$whereClause = !empty($whereClauses) ? 'WHERE ' . implode(' AND ', $whereClauses) : '';

// Get total count
$countQuery = "
    SELECT COUNT(*) 
    FROM affiliate_payouts p
    LEFT JOIN affiliate_partners ap ON p.partner_id = ap.id
    $whereClause
";
$countStmt = $db->prepare($countQuery);
$countStmt->execute($params);
$totalPayouts = $countStmt->fetchColumn();
$totalPages = ceil($totalPayouts / $perPage);

// Get payouts
$query = "
    SELECT p.*, ap.partner_name, ap.contact_email
    FROM affiliate_payouts p
    LEFT JOIN affiliate_partners ap ON p.partner_id = ap.id
    $whereClause
    ORDER BY p.created_at DESC
    LIMIT ? OFFSET ?
";
$stmt = $db->prepare($query);
$params[] = $perPage;
$params[] = $offset;
$stmt->execute($params);
$payouts = $stmt->fetchAll();

// Get partners for payout form
$partnersStmt = $db->prepare("
    SELECT id, partner_name, contact_email 
    FROM affiliate_partners 
    WHERE status = 'active' 
    ORDER BY partner_name
");
$partnersStmt->execute();
$partners = $partnersStmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-cash-coin"></i> Affiliate Payouts</h1>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#processPayoutModal">
                <i class="bi bi-plus-circle"></i> Process Payout
            </button>
        </div>
    </div>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= escape($error) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= escape($success) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Search and Filter -->
<div class="row mb-3">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-6">
                        <label for="search" class="form-label">Search</label>
                        <input type="text" class="form-control" id="search" name="search" 
                               value="<?= escape($search) ?>" placeholder="Partner name, email...">
                    </div>
                    <div class="col-md-4">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="all" <?= $statusFilter === 'all' ? 'selected' : '' ?>>All Statuses</option>
                            <option value="pending" <?= $statusFilter === 'pending' ? 'selected' : '' ?>>Pending</option>
                            <option value="processing" <?= $statusFilter === 'processing' ? 'selected' : '' ?>>Processing</option>
                            <option value="paid" <?= $statusFilter === 'paid' ? 'selected' : '' ?>>Paid</option>
                            <option value="failed" <?= $statusFilter === 'failed' ? 'selected' : '' ?>>Failed</option>
                            <option value="cancelled" <?= $statusFilter === 'cancelled' ? 'selected' : '' ?>>Cancelled</option>
                        </select>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Payouts Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Payouts (<?= number_format($totalPayouts) ?>)</h5>
            </div>
            <div class="card-body">
                <?php if (empty($payouts)): ?>
                    <p class="text-muted mb-0">No payouts found.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Partner</th>
                                    <th>Period</th>
                                    <th>Amount</th>
                                    <th>Conversions</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($payouts as $payout): ?>
                                    <tr>
                                        <td>
                                            <strong><?= escape($payout['partner_name']) ?></strong>
                                            <br><small class="text-muted"><?= escape($payout['contact_email']) ?></small>
                                        </td>
                                        <td>
                                            <?= formatDate($payout['payout_period_start']) ?> - 
                                            <?= formatDate($payout['payout_period_end']) ?>
                                        </td>
                                        <td>
                                            <strong>$<?= number_format($payout['total_commission'], 2) ?></strong>
                                            <br><small class="text-muted"><?= ucfirst($payout['payment_method']) ?></small>
                                        </td>
                                        <td><?= number_format($payout['conversion_count']) ?></td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $payout['payout_status'] === 'paid' ? 'success' : 
                                                ($payout['payout_status'] === 'processing' ? 'warning' : 
                                                ($payout['payout_status'] === 'failed' ? 'danger' : 'secondary'))
                                            ?>">
                                                <?= escape($payout['payout_status']) ?>
                                            </span>
                                        </td>
                                        <td><?= formatDate($payout['created_at']) ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <?php if ($payout['payout_status'] === 'pending'): ?>
                                                    <button class="btn btn-success" title="Mark as Paid" 
                                                            onclick="markAsPaid(<?= $payout['id'] ?>)">
                                                        <i class="bi bi-check-circle"></i>
                                                    </button>
                                                <?php endif; ?>
                                                <a href="<?= url('admin/payout_details.php?id=' . $payout['id']) ?>" 
                                                   class="btn btn-info" title="View Details">
                                                    <i class="bi bi-eye"></i>
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <nav aria-label="Page navigation">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>">Previous</a>
                                    </li>
                                <?php endif; ?>
                                
                                <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                    <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                        <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>"><?= $i ?></a>
                                    </li>
                                <?php endfor; ?>
                                
                                <?php if ($page < $totalPages): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>">Next</a>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Process Payout Modal -->
<div class="modal fade" id="processPayoutModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Process Payout</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="process_payout">
                    
                    <div class="mb-3">
                        <label for="partner_id" class="form-label">Partner <span class="text-danger">*</span></label>
                        <select class="form-select" id="partner_id" name="partner_id" required>
                            <option value="">Select Partner</option>
                            <?php foreach ($partners as $partner): ?>
                                <option value="<?= $partner['id'] ?>"><?= escape($partner['partner_name']) ?> (<?= escape($partner['contact_email']) ?>)</option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="payout_period_start" class="form-label">Period Start <span class="text-danger">*</span></label>
                            <input type="date" class="form-control" id="payout_period_start" name="payout_period_start" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="payout_period_end" class="form-label">Period End <span class="text-danger">*</span></label>
                            <input type="date" class="form-control" id="payout_period_end" name="payout_period_end" required>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="payment_method" class="form-label">Payment Method <span class="text-danger">*</span></label>
                        <select class="form-select" id="payment_method" name="payment_method" required>
                            <option value="paypal">PayPal</option>
                            <option value="bank_transfer">Bank Transfer</option>
                            <option value="check">Check</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="payment_reference" class="form-label">Payment Reference</label>
                        <input type="text" class="form-control" id="payment_reference" name="payment_reference" 
                               placeholder="Transaction ID, check number, etc.">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Process Payout</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Mark as Paid Modal -->
<div class="modal fade" id="markPaidModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Mark as Paid</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="mark_paid">
                    <input type="hidden" name="payout_id" id="paid_payout_id">
                    
                    <div class="mb-3">
                        <label for="payment_date" class="form-label">Payment Date</label>
                        <input type="date" class="form-control" id="payment_date" name="payment_date" 
                               value="<?= date('Y-m-d') ?>" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">Mark as Paid</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function markAsPaid(payoutId) {
    document.getElementById('paid_payout_id').value = payoutId;
    var modal = new bootstrap.Modal(document.getElementById('markPaidModal'));
    modal.show();
}
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
