<?php
/**
 * Admin - Partner Statistics
 * Detailed statistics for a specific affiliate partner
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/PartnerDashboard.php';

requireAdmin();
$pageTitle = 'Partner Statistics';

$db = getDB();
$partnerId = intval($_GET['id'] ?? 0);

if (!$partnerId) {
    setFlashMessage('danger', 'Invalid partner ID.');
    header('Location: ' . url('admin/affiliates.php'));
    exit;
}

// Get partner details
$stmt = $db->prepare("SELECT * FROM affiliate_partners WHERE id = ?");
$stmt->execute([$partnerId]);
$partner = $stmt->fetch();

if (!$partner) {
    setFlashMessage('danger', 'Partner not found.');
    header('Location: ' . url('admin/affiliates.php'));
    exit;
}

$dashboard = new PartnerDashboard();

// Get stats for different periods
$stats30 = $dashboard->getDashboardStats($partnerId, 30);
$stats90 = $dashboard->getDashboardStats($partnerId, 90);
$stats365 = $dashboard->getDashboardStats($partnerId, 365);

// Get recent conversions
$recentConversions = $dashboard->getRecentConversions($partnerId, 20);

// Get payout history
$payoutHistory = $dashboard->getPayoutHistory($partnerId, 10);

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-graph-up"></i> Partner Statistics</h1>
            <div class="d-flex gap-2">
                <a href="<?= url('admin/affiliates.php') ?>" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Back to Partners
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Partner Info -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Partner Information</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h6>Partner Details</h6>
                        <p><strong>Name:</strong> <?= escape($partner['partner_name']) ?></p>
                        <p><strong>Code:</strong> <code><?= escape($partner['partner_code']) ?></code></p>
                        <p><strong>Email:</strong> <?= escape($partner['contact_email']) ?></p>
                        <p><strong>Company:</strong> <?= escape($partner['company_name'] ?: 'N/A') ?></p>
                    </div>
                    <div class="col-md-6">
                        <h6>Commission Settings</h6>
                        <p><strong>Rate:</strong> <?= $partner['commission_rate'] ?>%</p>
                        <p><strong>Type:</strong> <?= ucfirst($partner['commission_type']) ?></p>
                        <p><strong>Status:</strong> 
                            <span class="badge bg-<?= $partner['status'] === 'active' ? 'success' : 'danger' ?>">
                                <?= escape($partner['status']) ?>
                            </span>
                        </p>
                        <p><strong>Referral Link:</strong> 
                            <a href="<?= url('refer.php?ref=' . $partner['partner_code']) ?>" target="_blank">
                                <?= url('refer.php?ref=' . $partner['partner_code']) ?>
                            </a>
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Stats Comparison -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Performance Comparison</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Period</th>
                                <th>Clicks</th>
                                <th>Conversions</th>
                                <th>Conversion Rate</th>
                                <th>Revenue</th>
                                <th>Commission</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td><strong>Last 30 Days</strong></td>
                                <td><?= number_format($stats30['clicks']['total']) ?></td>
                                <td><?= number_format($stats30['conversions']['total']) ?></td>
                                <td><?= $stats30['conversions']['rate'] ?>%</td>
                                <td>$<?= number_format($stats30['conversions']['revenue'], 2) ?></td>
                                <td>$<?= number_format($stats30['conversions']['commission'], 2) ?></td>
                            </tr>
                            <tr>
                                <td><strong>Last 90 Days</strong></td>
                                <td><?= number_format($stats90['clicks']['total']) ?></td>
                                <td><?= number_format($stats90['conversions']['total']) ?></td>
                                <td><?= $stats90['conversions']['rate'] ?>%</td>
                                <td>$<?= number_format($stats90['conversions']['revenue'], 2) ?></td>
                                <td>$<?= number_format($stats90['conversions']['commission'], 2) ?></td>
                            </tr>
                            <tr>
                                <td><strong>Last 365 Days</strong></td>
                                <td><?= number_format($stats365['clicks']['total']) ?></td>
                                <td><?= number_format($stats365['conversions']['total']) ?></td>
                                <td><?= $stats365['conversions']['rate'] ?>%</td>
                                <td>$<?= number_format($stats365['conversions']['revenue'], 2) ?></td>
                                <td>$<?= number_format($stats365['conversions']['commission'], 2) ?></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Recent Conversions -->
<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Recent Conversions</h5>
            </div>
            <div class="card-body">
                <?php if (empty($recentConversions)): ?>
                    <p class="text-muted mb-0">No conversions found</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Client</th>
                                    <th>Type</th>
                                    <th>Value</th>
                                    <th>Commission</th>
                                    <th>Status</th>
                                    <th>Date</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recentConversions as $conversion): ?>
                                    <tr>
                                        <td>
                                            <strong><?= escape($conversion['client_email'] ?? 'Unknown') ?></strong>
                                            <?php if ($conversion['business_name']): ?>
                                                <br><small class="text-muted"><?= escape($conversion['business_name']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= escape($conversion['conversion_type']) ?></td>
                                        <td>$<?= number_format($conversion['conversion_value'], 2) ?></td>
                                        <td>$<?= number_format($conversion['commission_amount'], 2) ?></td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $conversion['commission_status'] === 'paid' ? 'success' : 
                                                ($conversion['commission_status'] === 'approved' ? 'primary' : 'warning')
                                            ?>">
                                                <?= escape($conversion['commission_status']) ?>
                                            </span>
                                        </td>
                                        <td><?= formatDate($conversion['converted_at']) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Payout History</h5>
            </div>
            <div class="card-body">
                <?php if (empty($payoutHistory)): ?>
                    <p class="text-muted mb-0">No payouts yet</p>
                <?php else: ?>
                    <div class="list-group list-group-flush">
                        <?php foreach ($payoutHistory as $payout): ?>
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                <div>
                                    <strong>$<?= number_format($payout['total_commission'], 2) ?></strong>
                                    <br><small class="text-muted"><?= formatDate($payout['payout_period_start']) ?> - <?= formatDate($payout['payout_period_end']) ?></small>
                                </div>
                                <div class="text-end">
                                    <span class="badge bg-<?= $payout['payout_status'] === 'paid' ? 'success' : 'warning' ?>">
                                        <?= escape($payout['payout_status']) ?>
                                    </span>
                                    <br><small class="text-muted"><?= formatDate($payout['created_at']) ?></small>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
