<?php
/**
 * Admin - Lead Management
 * Manage client leads from affiliate referrals
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/csrf.php';

requireAdmin();
$pageTitle = 'Lead Management';

$db = getDB();
$error = '';
$success = '';

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Search and Filter
$search = trim($_GET['search'] ?? '');
$statusFilter = $_GET['status'] ?? 'all';
$sourceFilter = $_GET['source'] ?? 'all';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requireCSRFToken();
    
    $action = $_POST['action'] ?? '';
    $leadId = intval($_POST['lead_id'] ?? 0);
    
    if ($action === 'update_status' && $leadId > 0) {
        $newStatus = $_POST['new_status'] ?? '';
        $notes = trim($_POST['notes'] ?? '');
        
        $validStatuses = ['new', 'contacted', 'qualified', 'converted', 'lost'];
        if (in_array($newStatus, $validStatuses)) {
            $stmt = $db->prepare("
                UPDATE client_leads 
                SET status = ?, notes = COALESCE(?, notes), updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$newStatus, $notes, $leadId]);
            $success = 'Lead status updated successfully';
        } else {
            $error = 'Invalid status';
        }
    }
}

// Build query with filters
$whereClauses = [];
$params = [];

if (!empty($search)) {
    $whereClauses[] = "(lead_email LIKE ? OR lead_name LIKE ? OR company_name LIKE ?)";
    $searchTerm = '%' . $search . '%';
    $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm]);
}

if ($statusFilter !== 'all') {
    $whereClauses[] = "cl.status = ?";
    $params[] = $statusFilter;
}

if ($sourceFilter !== 'all') {
    $whereClauses[] = "cl.lead_source = ?";
    $params[] = $sourceFilter;
}

$whereClause = !empty($whereClauses) ? 'WHERE ' . implode(' AND ', $whereClauses) : '';

// Get total count
$countQuery = "
    SELECT COUNT(*) 
    FROM client_leads cl
    LEFT JOIN affiliate_partners ap ON cl.partner_id = ap.id
    $whereClause
";
$countStmt = $db->prepare($countQuery);
$countStmt->execute($params);
$totalLeads = $countStmt->fetchColumn();
$totalPages = ceil($totalLeads / $perPage);

// Get leads
$query = "
    SELECT cl.*, ap.partner_name, ap.partner_code, u.business_name as converted_business
    FROM client_leads cl
    LEFT JOIN affiliate_partners ap ON cl.partner_id = ap.id
    LEFT JOIN users u ON cl.converted_to_user_id = u.id
    $whereClause
    ORDER BY cl.created_at DESC
    LIMIT ? OFFSET ?
";
$stmt = $db->prepare($query);
$params[] = $perPage;
$params[] = $offset;
$stmt->execute($params);
$leads = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-person-lines-fill"></i> Client Leads</h1>
            <div class="d-flex gap-2">
                <a href="<?= url('admin/leads_export.php') ?>" class="btn btn-outline-success">
                    <i class="bi bi-download"></i> Export CSV
                </a>
            </div>
        </div>
    </div>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= escape($error) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= escape($success) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Search and Filter -->
<div class="row mb-3">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-4">
                        <label for="search" class="form-label">Search</label>
                        <input type="text" class="form-control" id="search" name="search" 
                               value="<?= escape($search) ?>" placeholder="Email, name, company...">
                    </div>
                    <div class="col-md-3">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="all" <?= $statusFilter === 'all' ? 'selected' : '' ?>>All Statuses</option>
                            <option value="new" <?= $statusFilter === 'new' ? 'selected' : '' ?>>New</option>
                            <option value="contacted" <?= $statusFilter === 'contacted' ? 'selected' : '' ?>>Contacted</option>
                            <option value="qualified" <?= $statusFilter === 'qualified' ? 'selected' : '' ?>>Qualified</option>
                            <option value="converted" <?= $statusFilter === 'converted' ? 'selected' : '' ?>>Converted</option>
                            <option value="lost" <?= $statusFilter === 'lost' ? 'selected' : '' ?>>Lost</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="source" class="form-label">Source</label>
                        <select class="form-select" id="source" name="source">
                            <option value="all" <?= $sourceFilter === 'all' ? 'selected' : '' ?>>All Sources</option>
                            <option value="affiliate" <?= $sourceFilter === 'affiliate' ? 'selected' : '' ?>>Affiliate</option>
                            <option value="direct" <?= $sourceFilter === 'direct' ? 'selected' : '' ?>>Direct</option>
                            <option value="organic" <?= $sourceFilter === 'organic' ? 'selected' : '' ?>>Organic</option>
                            <option value="paid_ads" <?= $sourceFilter === 'paid_ads' ? 'selected' : '' ?>>Paid Ads</option>
                        </select>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Leads Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Leads (<?= number_format($totalLeads) ?>)</h5>
            </div>
            <div class="card-body">
                <?php if (empty($leads)): ?>
                    <p class="text-muted mb-0">No leads found.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Lead Info</th>
                                    <th>Source</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Converted</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($leads as $lead): ?>
                                    <tr>
                                        <td>
                                            <strong><?= escape($lead['lead_name'] ?: 'Unknown') ?></strong>
                                            <br><small class="text-muted"><?= escape($lead['lead_email']) ?></small>
                                            <?php if ($lead['company_name']): ?>
                                                <br><small class="text-muted"><?= escape($lead['company_name']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($lead['partner_name']): ?>
                                                <span class="badge bg-info"><?= escape($lead['partner_name']) ?></span>
                                                <br><small class="text-muted">Code: <?= escape($lead['partner_code']) ?></small>
                                            <?php else: ?>
                                                <span class="badge bg-secondary"><?= ucfirst($lead['lead_source']) ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $lead['status'] === 'converted' ? 'success' : 
                                                ($lead['status'] === 'qualified' ? 'primary' : 
                                                ($lead['status'] === 'contacted' ? 'warning' : 
                                                ($lead['status'] === 'lost' ? 'danger' : 'secondary')))
                                            ?>">
                                                <?= escape($lead['status']) ?>
                                            </span>
                                        </td>
                                        <td><?= formatDate($lead['created_at']) ?></td>
                                        <td>
                                            <?php if ($lead['converted_at']): ?>
                                                <?= formatDate($lead['converted_at']) ?>
                                                <?php if ($lead['converted_business']): ?>
                                                    <br><small class="text-muted"><?= escape($lead['converted_business']) ?></small>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <span class="text-muted">Not converted</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <button class="btn btn-primary" title="Update Status" 
                                                        onclick="updateLeadStatus(<?= $lead['id'] ?>, '<?= escape($lead['status']) ?>', '<?= escape($lead['notes'] ?? '') ?>')">
                                                    <i class="bi bi-pencil"></i>
                                                </button>
                                                <?php if ($lead['status'] === 'converted' && $lead['converted_to_user_id']): ?>
                                                    <a href="<?= url('admin/view_client.php?client_id=' . $lead['converted_to_user_id']) ?>" 
                                                       class="btn btn-success" title="View Client">
                                                        <i class="bi bi-person"></i>
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <nav aria-label="Page navigation">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>&source=<?= $sourceFilter ?>">Previous</a>
                                    </li>
                                <?php endif; ?>
                                
                                <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                    <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                        <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>&source=<?= $sourceFilter ?>"><?= $i ?></a>
                                    </li>
                                <?php endfor; ?>
                                
                                <?php if ($page < $totalPages): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>&source=<?= $sourceFilter ?>">Next</a>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Update Status Modal -->
<div class="modal fade" id="updateStatusModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Update Lead Status</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="update_status">
                    <input type="hidden" name="lead_id" id="lead_id">
                    
                    <div class="mb-3">
                        <label for="new_status" class="form-label">Status</label>
                        <select class="form-select" id="new_status" name="new_status" required>
                            <option value="new">New</option>
                            <option value="contacted">Contacted</option>
                            <option value="qualified">Qualified</option>
                            <option value="converted">Converted</option>
                            <option value="lost">Lost</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="notes" class="form-label">Notes</label>
                        <textarea class="form-control" id="notes" name="notes" rows="3" 
                                  placeholder="Add notes about this lead..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Status</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function updateLeadStatus(leadId, currentStatus, currentNotes) {
    document.getElementById('lead_id').value = leadId;
    document.getElementById('new_status').value = currentStatus;
    document.getElementById('notes').value = currentNotes;
    
    var modal = new bootstrap.Modal(document.getElementById('updateStatusModal'));
    modal.show();
}
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
