<?php
/**
 * Admin - Export Clients to CSV
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireAdmin();

$db = getDB();

// Get search and filter parameters
$search = trim($_GET['search'] ?? '');
$statusFilter = $_GET['status'] ?? 'all';
$subscriptionFilter = $_GET['subscription'] ?? 'all';
$exportIds = $_GET['ids'] ?? null; // Optional: export specific IDs

// Build query
$whereClauses = ["u.role = 'client'"];
$params = [];

if (!empty($search)) {
    $whereClauses[] = "(u.business_name LIKE ? OR u.email LIKE ? OR u.contact_name LIKE ?)";
    $searchTerm = '%' . $search . '%';
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

if ($statusFilter !== 'all') {
    $whereClauses[] = "u.status = ?";
    $params[] = $statusFilter;
}

if ($subscriptionFilter === 'with') {
    $whereClauses[] = "s.status = 'active'";
} elseif ($subscriptionFilter === 'without') {
    $whereClauses[] = "s.status IS NULL";
}

// If specific IDs provided
if ($exportIds) {
    $ids = array_filter(array_map('intval', explode(',', $exportIds)));
    if (!empty($ids)) {
        $placeholders = str_repeat('?,', count($ids) - 1) . '?';
        $whereClauses[] = "u.id IN ($placeholders)";
        $params = array_merge($params, $ids);
    }
}

$whereClause = implode(' AND ', $whereClauses);

// Get clients
$query = "
    SELECT DISTINCT u.*, s.status as subscription_status, s.renewal_date,
           st.tier_name, st.tier_level
    FROM users u
    LEFT JOIN subscriptions s ON u.id = s.user_id AND s.status = 'active'
    LEFT JOIN subscription_tiers st ON s.tier_id = st.id
    WHERE $whereClause
    ORDER BY u.created_at DESC
";
$stmt = $db->prepare($query);
$stmt->execute($params);
$clients = $stmt->fetchAll();

// Set headers for CSV download
$filename = 'clients_export_' . date('Y-m-d_His') . '.csv';
header('Content-Type: text/csv');
header('Content-Disposition: attachment; filename="' . $filename . '"');
header('Pragma: no-cache');
header('Expires: 0');

// Open output stream
$output = fopen('php://output', 'w');

// Add CSV header
fputcsv($output, [
    'ID',
    'Email',
    'Business Name',
    'Contact Name',
    'Phone',
    'Address',
    'City',
    'State',
    'Zip Code',
    'Status',
    'Subscription Tier',
    'Subscription Status',
    'Renewal Date',
    'Created At',
    'Updated At'
]);

// Add data rows
foreach ($clients as $client) {
    fputcsv($output, [
        $client['id'],
        $client['email'],
        $client['business_name'],
        $client['contact_name'],
        $client['phone'] ?? '',
        $client['address'] ?? '',
        $client['city'] ?? '',
        $client['state'] ?? '',
        $client['zip_code'] ?? '',
        $client['status'],
        $client['tier_name'] ?? 'No Subscription',
        $client['subscription_status'] ?? 'None',
        $client['renewal_date'] ?? '',
        $client['created_at'],
        $client['updated_at']
    ]);
}

fclose($output);
exit;

