<?php
/**
 * Admin - Client Bulk Actions Handler
 * Handles bulk operations: delete, activate, deactivate, export
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/activity_log.php';

requireAdmin();

$db = getDB();

// Handle bulk actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requireCSRFToken();
    
    $action = $_POST['bulk_action'] ?? '';
    $clientIds = $_POST['client_ids'] ?? [];
    
    if (empty($action) || empty($clientIds) || !is_array($clientIds)) {
        setFlashMessage('danger', 'No clients selected or invalid action');
        header('Location: ' . url('admin/clients.php'));
        exit;
    }
    
    // Sanitize client IDs
    $clientIds = array_filter(array_map('intval', $clientIds));
    
    if (empty($clientIds)) {
        setFlashMessage('danger', 'No valid clients selected');
        header('Location: ' . url('admin/clients.php'));
        exit;
    }
    
    $placeholders = str_repeat('?,', count($clientIds) - 1) . '?';
    
    try {
        switch ($action) {
            case 'activate':
                $stmt = $db->prepare("
                    UPDATE users 
                    SET status = 'active', updated_at = NOW() 
                    WHERE id IN ($placeholders) AND role = 'client'
                ");
                $stmt->execute($clientIds);
                logActivity('bulk_action', 'user', null, "Bulk activated " . count($clientIds) . " client(s)", null, null, count($clientIds));
                setFlashMessage('success', count($clientIds) . ' client(s) activated successfully!');
                break;
                
            case 'deactivate':
                $stmt = $db->prepare("
                    UPDATE users 
                    SET status = 'inactive', updated_at = NOW() 
                    WHERE id IN ($placeholders) AND role = 'client'
                ");
                $stmt->execute($clientIds);
                logActivity('bulk_action', 'user', null, "Bulk deactivated " . count($clientIds) . " client(s)", null, null, count($clientIds));
                setFlashMessage('success', count($clientIds) . ' client(s) deactivated successfully!');
                break;
                
            case 'suspend':
                $stmt = $db->prepare("
                    UPDATE users 
                    SET status = 'suspended', updated_at = NOW()
                    WHERE id IN ($placeholders) AND role = 'client'
                ");
                $stmt->execute($clientIds);
                logActivity('bulk_action', 'user', null, "Bulk suspended " . count($clientIds) . " client(s)", null, null, count($clientIds));
                setFlashMessage('success', count($clientIds) . ' client(s) suspended successfully!');
                break;
                
            default:
                setFlashMessage('danger', 'Invalid bulk action');
        }
    } catch (PDOException $e) {
        setFlashMessage('danger', 'Error performing bulk action: ' . $e->getMessage());
    }
    
    header('Location: ' . url('admin/clients.php'));
    exit;
}

header('Location: /admin/clients.php');
exit;

