<?php
/**
 * Admin - Client Management
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireAdmin();
$pageTitle = 'Client Management';

$db = getDB();

// Handle form submission
$error = '';
$success = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'create_user') {
    $formData['email'] = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $formData['business_name'] = trim($_POST['business_name'] ?? '');
    $formData['contact_name'] = trim($_POST['contact_name'] ?? '');
    $formData['phone'] = trim($_POST['phone'] ?? '');
    $formData['address'] = trim($_POST['address'] ?? '');
    $formData['city'] = trim($_POST['city'] ?? '');
    $formData['state'] = trim($_POST['state'] ?? '');
    $formData['zip_code'] = trim($_POST['zip_code'] ?? '');
    
    $email = $formData['email'];
    $businessName = $formData['business_name'];
    $contactName = $formData['contact_name'];
    $phone = $formData['phone'];
    $address = $formData['address'];
    $city = $formData['city'];
    $state = $formData['state'];
    $zipCode = $formData['zip_code'];
    
    // Validation
    if (empty($email) || !isValidEmail($email)) {
        $error = 'Valid email address is required';
    } elseif (empty($password) || strlen($password) < 8) {
        $error = 'Password must be at least 8 characters long';
    } elseif (empty($businessName)) {
        $error = 'Business name is required';
    } elseif (empty($contactName)) {
        $error = 'Contact name is required';
    } else {
        // Check if email already exists
        $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetchColumn() > 0) {
            $error = 'Email address already exists';
        } else {
            try {
                // Create user
                $passwordHash = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $db->prepare("
                    INSERT INTO users (email, password_hash, business_name, contact_name, phone, address, city, state, zip_code, role, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'client', 'active')
                ");
                $stmt->execute([$email, $passwordHash, $businessName, $contactName, $phone ?: null, $address ?: null, $city ?: null, $state ?: null, $zipCode ?: null]);
                
                setFlashMessage('success', 'Client created successfully!');
                header('Location: /admin/clients.php');
                exit;
            } catch (PDOException $e) {
                $error = 'Error creating user: ' . $e->getMessage();
            }
        }
    }
}

// Get all clients with their subscription info
$stmt = $db->query("
    SELECT u.*, s.id as subscription_id, s.status as subscription_status, s.renewal_date,
           st.tier_name, st.tier_level
    FROM users u
    LEFT JOIN subscriptions s ON u.id = s.user_id AND s.status = 'active'
    LEFT JOIN subscription_tiers st ON s.tier_id = st.id
    WHERE u.role = 'client'
    ORDER BY u.created_at DESC
");
$clients = $stmt->fetchAll();

$flash = getFlashMessage();
require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <h1>Client Management</h1>
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createUserModal">
            <i class="bi bi-plus-circle"></i> Add New Client
        </button>
    </div>
</div>

<?php if ($flash): ?>
    <div class="alert alert-<?= escape($flash['type']) ?> alert-dismissible fade show" role="alert">
        <?= escape($flash['message']) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= escape($error) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">All Clients (<?= count($clients) ?>)</h5>
            </div>
            <div class="card-body">
                <?php if (empty($clients)): ?>
                    <p class="text-muted mb-0">No clients found. Click "Add New Client" to create one.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Business Name</th>
                                <th>Contact</th>
                                <th>Email</th>
                                <th>Subscription</th>
                                <th>Status</th>
                                <th>Joined</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($clients as $client): ?>
                                <tr>
                                    <td><strong><?= escape($client['business_name']) ?></strong></td>
                                    <td><?= escape($client['contact_name']) ?></td>
                                    <td><?= escape($client['email']) ?></td>
                                    <td>
                                        <?php if ($client['subscription_status'] === 'active'): ?>
                                            <span class="badge bg-success"><?= escape($client['tier_name'] ?? 'No Plan') ?></span>
                                            <br><small class="text-muted">Renewal: <?= formatDate($client['renewal_date']) ?></small>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">No Active Subscription</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?= $client['status'] === 'active' ? 'success' : 'secondary' ?>">
                                            <?= escape($client['status']) ?>
                                        </span>
                                    </td>
                                    <td><?= formatDate($client['created_at']) ?></td>
                                    <td>
                                        <a href="/admin/subscription_manage.php?client_id=<?= $client['id'] ?>" class="btn btn-sm btn-primary">
                                            Manage Subscription
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Create User Modal -->
<div class="modal fade" id="createUserModal" tabindex="-1" aria-labelledby="createUserModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="createUserModalLabel">Add New Client</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" action="/admin/clients.php">
                <div class="modal-body">
                    <input type="hidden" name="action" value="create_user">
                    
                    <?php if ($error): ?>
                        <div class="alert alert-danger">
                            <?= escape($error) ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="email" class="form-label">Email Address <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" id="email" name="email" value="<?= escape($formData['email'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="password" class="form-label">Password <span class="text-danger">*</span></label>
                            <input type="password" class="form-control" id="password" name="password" required minlength="8">
                            <small class="form-text text-muted">Minimum 8 characters</small>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="business_name" class="form-label">Business Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="business_name" name="business_name" value="<?= escape($formData['business_name'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="contact_name" class="form-label">Contact Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="contact_name" name="contact_name" value="<?= escape($formData['contact_name'] ?? '') ?>" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="phone" class="form-label">Phone</label>
                            <input type="tel" class="form-control" id="phone" name="phone" value="<?= escape($formData['phone'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="zip_code" class="form-label">Zip Code</label>
                            <input type="text" class="form-control" id="zip_code" name="zip_code" value="<?= escape($formData['zip_code'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="address" class="form-label">Address</label>
                        <input type="text" class="form-control" id="address" name="address" value="<?= escape($formData['address'] ?? '') ?>">
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="city" class="form-label">City</label>
                            <input type="text" class="form-control" id="city" name="city" value="<?= escape($formData['city'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="state" class="form-label">State</label>
                            <input type="text" class="form-control" id="state" name="state" value="<?= escape($formData['state'] ?? '') ?>">
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Client</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php if ($error): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var modal = new bootstrap.Modal(document.getElementById('createUserModal'));
    modal.show();
});
</script>
<?php endif; ?>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>


