<?php
/**
 * Admin - Enhanced Client Management
 * Features: Create, Edit, Delete, Search, Filter, Pagination
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/activity_log.php';

requireAdmin();
$pageTitle = 'Client Management';

$db = getDB();
$error = '';
$formData = [];

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Search and Filter
$search = trim($_GET['search'] ?? '');
$statusFilter = $_GET['status'] ?? 'all';
$subscriptionFilter = $_GET['subscription'] ?? 'all';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requireCSRFToken();
    
    $action = $_POST['action'] ?? '';
    $clientId = isset($_POST['client_id']) ? intval($_POST['client_id']) : 0;
    
    if ($action === 'create_user') {
        $formData['email'] = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $formData['business_name'] = trim($_POST['business_name'] ?? '');
        $formData['contact_name'] = trim($_POST['contact_name'] ?? '');
        $formData['phone'] = trim($_POST['phone'] ?? '');
        $formData['address'] = trim($_POST['address'] ?? '');
        $formData['city'] = trim($_POST['city'] ?? '');
        $formData['state'] = trim($_POST['state'] ?? '');
        $formData['zip_code'] = trim($_POST['zip_code'] ?? '');
        
        $email = $formData['email'];
        $businessName = $formData['business_name'];
        $contactName = $formData['contact_name'];
        
        if (empty($email) || !isValidEmail($email)) {
            $error = 'Valid email address is required';
        } elseif (empty($password) || strlen($password) < 8) {
            $error = 'Password must be at least 8 characters long';
        } elseif (empty($businessName)) {
            $error = 'Business name is required';
        } elseif (empty($contactName)) {
            $error = 'Contact name is required';
        } else {
            $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE email = ?");
            $stmt->execute([$email]);
            if ($stmt->fetchColumn() > 0) {
                $error = 'Email address already exists';
            } else {
                try {
                    $passwordHash = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $db->prepare("
                        INSERT INTO users (email, password_hash, business_name, contact_name, phone, address, city, state, zip_code, role, status)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'client', 'active')
                    ");
                    $stmt->execute([$email, $passwordHash, $businessName, $contactName, 
                        $formData['phone'] ?: null, $formData['address'] ?: null, 
                        $formData['city'] ?: null, $formData['state'] ?: null, 
                        $formData['zip_code'] ?: null]);
                    
                    $newUserId = $db->lastInsertId();
                    
                    // Automatically assign this client to the creating admin
                    assignClientToAdmin($_SESSION['user_id'], $newUserId, $_SESSION['user_id']);
                    
                    logActivity('create', 'user', $newUserId, "Created new client: {$businessName} ({$email})");
                    
                    setFlashMessage('success', 'Client created successfully!');
                    header('Location: ' . url('admin/clients.php'));
                    exit;
                } catch (PDOException $e) {
                    $error = 'Error creating user: ' . $e->getMessage();
                }
            }
        }
        
    } elseif ($action === 'update_user' && $clientId > 0) {
        $formData['email'] = trim($_POST['email'] ?? '');
        $formData['business_name'] = trim($_POST['business_name'] ?? '');
        $formData['contact_name'] = trim($_POST['contact_name'] ?? '');
        $formData['phone'] = trim($_POST['phone'] ?? '');
        $formData['address'] = trim($_POST['address'] ?? '');
        $formData['city'] = trim($_POST['city'] ?? '');
        $formData['state'] = trim($_POST['state'] ?? '');
        $formData['zip_code'] = trim($_POST['zip_code'] ?? '');
        $status = $_POST['status'] ?? 'active';
        $password = trim($_POST['password'] ?? '');
        
        $email = $formData['email'];
        $businessName = $formData['business_name'];
        $contactName = $formData['contact_name'];
        
        if (empty($email) || !isValidEmail($email)) {
            $error = 'Valid email address is required';
        } elseif (empty($businessName)) {
            $error = 'Business name is required';
        } elseif (empty($contactName)) {
            $error = 'Contact name is required';
        } elseif (!empty($password) && strlen($password) < 8) {
            $error = 'Password must be at least 8 characters long';
        } else {
            // Check if email already exists (excluding current user)
            $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE email = ? AND id != ?");
            $stmt->execute([$email, $clientId]);
            if ($stmt->fetchColumn() > 0) {
                $error = 'Email address already exists';
            } else {
                try {
                    if (!empty($password)) {
                        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
                        $stmt = $db->prepare("
                            UPDATE users SET email = ?, password_hash = ?, business_name = ?, contact_name = ?, 
                            phone = ?, address = ?, city = ?, state = ?, zip_code = ?, status = ?, updated_at = NOW()
                            WHERE id = ? AND role = 'client'
                        ");
                        $stmt->execute([$email, $passwordHash, $businessName, $contactName,
                            $formData['phone'] ?: null, $formData['address'] ?: null,
                            $formData['city'] ?: null, $formData['state'] ?: null,
                            $formData['zip_code'] ?: null, $status, $clientId]);
                    } else {
                        $stmt = $db->prepare("
                            UPDATE users SET email = ?, business_name = ?, contact_name = ?, 
                            phone = ?, address = ?, city = ?, state = ?, zip_code = ?, status = ?, updated_at = NOW()
                            WHERE id = ? AND role = 'client'
                        ");
                        $stmt->execute([$email, $businessName, $contactName,
                            $formData['phone'] ?: null, $formData['address'] ?: null,
                            $formData['city'] ?: null, $formData['state'] ?: null,
                            $formData['zip_code'] ?: null, $status, $clientId]);
                    }
                    
                    logActivity('update', 'user', $clientId, "Updated client: {$businessName} (Status: {$status})", 
                        null, ['email' => $email, 'business_name' => $businessName, 'status' => $status]);
                    
                    setFlashMessage('success', 'Client updated successfully!');
                    header('Location: ' . url('admin/clients.php'));
                    exit;
                } catch (PDOException $e) {
                    $error = 'Error updating user: ' . $e->getMessage();
                }
            }
        }
        
    } elseif ($action === 'delete_user' && $clientId > 0) {
        try {
            // Get client info for logging
            $stmt = $db->prepare("SELECT business_name, email FROM users WHERE id = ? AND role = 'client'");
            $stmt->execute([$clientId]);
            $clientInfo = $stmt->fetch();
            
            // Don't actually delete, just deactivate
            $stmt = $db->prepare("UPDATE users SET status = 'inactive', updated_at = NOW() WHERE id = ? AND role = 'client'");
            $stmt->execute([$clientId]);
            
            if ($clientInfo) {
                logActivity('deactivate', 'user', $clientId, "Deactivated client: {$clientInfo['business_name']} ({$clientInfo['email']})");
            }
            
            setFlashMessage('success', 'Client deactivated successfully!');
            header('Location: ' . url('admin/clients.php'));
            exit;
        } catch (PDOException $e) {
            $error = 'Error deactivating user: ' . $e->getMessage();
        }
    }
}

// Build query with filters - only show clients assigned to this admin
$adminId = $_SESSION['user_id'];
$adminClientIds = getAdminClients($adminId);

// Build query with filters
if (empty($adminClientIds)) {
    // Admin has no assigned clients, return empty result
    $whereClauses = ["u.role = 'client'", "1 = 0"]; // Always false condition
    $params = [];
} else {
    $whereClauses = ["u.role = 'client'", "u.id IN (" . str_repeat('?,', count($adminClientIds) - 1) . "?)"];
    $params = $adminClientIds;
}

if (!empty($search)) {
    $whereClauses[] = "(u.business_name LIKE ? OR u.email LIKE ? OR u.contact_name LIKE ?)";
    $searchTerm = '%' . $search . '%';
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

if ($statusFilter !== 'all') {
    $whereClauses[] = "u.status = ?";
    $params[] = $statusFilter;
}

if ($subscriptionFilter === 'with') {
    $whereClauses[] = "s.status = 'active'";
} elseif ($subscriptionFilter === 'without') {
    $whereClauses[] = "s.status IS NULL";
}

$whereClause = implode(' AND ', $whereClauses);

// Get total count for pagination
$countParams = $params;
$countQuery = "
    SELECT COUNT(DISTINCT u.id)
    FROM users u
    LEFT JOIN subscriptions s ON u.id = s.user_id AND s.status = 'active'
    LEFT JOIN subscription_tiers st ON s.tier_id = st.id
    WHERE $whereClause
";
$countStmt = $db->prepare($countQuery);
$countStmt->execute($countParams);
$totalClients = $countStmt->fetchColumn();
$totalPages = ceil($totalClients / $perPage);

// Get clients with pagination
$queryParams = $params;
$query = "
    SELECT DISTINCT u.*, s.id as subscription_id, s.status as subscription_status, s.renewal_date,
           st.tier_name, st.tier_level
    FROM users u
    LEFT JOIN subscriptions s ON u.id = s.user_id AND s.status = 'active'
    LEFT JOIN subscription_tiers st ON s.tier_id = st.id
    WHERE $whereClause
    ORDER BY u.created_at DESC
    LIMIT ? OFFSET ?
";
$stmt = $db->prepare($query);
$queryParams[] = $perPage;
$queryParams[] = $offset;
$stmt->execute($queryParams);
$clients = $stmt->fetchAll();

// Get client for editing
$editingClient = null;
if (isset($_GET['edit']) && intval($_GET['edit']) > 0) {
    $editId = intval($_GET['edit']);
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ? AND role = 'client'");
    $stmt->execute([$editId]);
    $editingClient = $stmt->fetch();
    if ($editingClient) {
        $formData = $editingClient;
    }
}

$flash = getFlashMessage();
require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <h1>Client Management</h1>
        <div class="d-flex gap-2">
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createUserModal">
                <i class="bi bi-plus-circle"></i> Add New Client
            </button>
            <?php if ($editingClient): ?>
                <button class="btn btn-secondary" onclick="window.location.href='/admin/clients.php'">
                    Cancel Editing
                </button>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php if ($flash): ?>
    <div class="alert alert-<?= escape($flash['type']) ?> alert-dismissible fade show" role="alert">
        <?= escape($flash['message']) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= escape($error) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Search and Filter -->
<div class="row mb-3">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <form method="GET" action="<?= url('admin/clients.php') ?>" class="row g-3">
                    <div class="col-md-4">
                        <label for="search" class="form-label">Search</label>
                        <input type="text" class="form-control" id="search" name="search" 
                               value="<?= escape($search) ?>" placeholder="Business name, email, contact...">
                    </div>
                    <div class="col-md-3">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="all" <?= $statusFilter === 'all' ? 'selected' : '' ?>>All Statuses</option>
                            <option value="active" <?= $statusFilter === 'active' ? 'selected' : '' ?>>Active</option>
                            <option value="inactive" <?= $statusFilter === 'inactive' ? 'selected' : '' ?>>Inactive</option>
                            <option value="suspended" <?= $statusFilter === 'suspended' ? 'selected' : '' ?>>Suspended</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="subscription" class="form-label">Subscription</label>
                        <select class="form-select" id="subscription" name="subscription">
                            <option value="all" <?= $subscriptionFilter === 'all' ? 'selected' : '' ?>>All</option>
                            <option value="with" <?= $subscriptionFilter === 'with' ? 'selected' : '' ?>>With Active Subscription</option>
                            <option value="without" <?= $subscriptionFilter === 'without' ? 'selected' : '' ?>>Without Subscription</option>
                        </select>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Clients Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">All Clients (<?= number_format($totalClients) ?>)</h5>
                <div class="d-flex align-items-center gap-2">
                    <small class="text-muted">Page <?= $page ?> of <?= $totalPages ?></small>
                    <a href="<?= url('admin/clients_export.php?' . http_build_query(['search' => $search, 'status' => $statusFilter, 'subscription' => $subscriptionFilter])) ?>" 
                       class="btn btn-sm btn-outline-success">
                        <i class="bi bi-download"></i> Export CSV
                    </a>
                </div>
            </div>
            <div class="card-body">
                <?php if (empty($clients)): ?>
                    <p class="text-muted mb-0">No clients found. Click "Add New Client" to create one.</p>
                <?php else: ?>
                    <!-- Bulk Actions -->
                    <div class="mb-3 d-flex gap-2 align-items-center">
                        <form method="POST" action="<?= url('admin/clients_bulk_actions.php') ?>" id="bulkActionForm" class="d-flex gap-2 flex-grow-1">
                            <?= csrfField() ?>
                            <select name="bulk_action" id="bulk_action" class="form-select" style="max-width: 200px;" required>
                                <option value="">Bulk Actions...</option>
                                <option value="activate">Activate Selected</option>
                                <option value="deactivate">Deactivate Selected</option>
                                <option value="suspend">Suspend Selected</option>
                            </select>
                            <button type="submit" class="btn btn-sm btn-primary" id="bulkActionBtn" disabled>
                                Apply
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="selectAll()">Select All</button>
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="selectNone()">Deselect All</button>
                        </form>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th width="40">
                                        <input type="checkbox" id="selectAllCheckbox" onchange="toggleAll(this)">
                                    </th>
                                    <th>Business Name</th>
                                    <th>Contact</th>
                                    <th>Email</th>
                                    <th>Subscription</th>
                                    <th>Status</th>
                                    <th>Joined</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($clients as $client): ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" name="client_ids[]" class="client-checkbox" 
                                                   value="<?= $client['id'] ?>" onchange="updateBulkButton()">
                                        </td>
                                        <td><strong><?= escape($client['business_name']) ?></strong></td>
                                        <td><?= escape($client['contact_name']) ?></td>
                                        <td><?= escape($client['email']) ?></td>
                                        <td>
                                            <?php if ($client['subscription_status'] === 'active'): ?>
                                                <span class="badge bg-success"><?= escape($client['tier_name'] ?? 'No Plan') ?></span>
                                                <br><small class="text-muted">Renewal: <?= formatDate($client['renewal_date']) ?></small>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">No Active Subscription</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?= $client['status'] === 'active' ? 'success' : ($client['status'] === 'suspended' ? 'danger' : 'secondary') ?>">
                                                <?= escape($client['status']) ?>
                                            </span>
                                        </td>
                                        <td><?= formatDate($client['created_at']) ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm" role="group">
                                                <a href="<?= url('admin/view_client.php?client_id=' . $client['id']) ?>" 
                                                   class="btn btn-success" title="View Client Dashboard">
                                                    <i class="bi bi-eye"></i>
                                                </a>
                                                <a href="<?= url('admin/subscription_manage.php?client_id=' . $client['id']) ?>" 
                                                   class="btn btn-primary" title="Manage Subscription">
                                                    <i class="bi bi-credit-card"></i>
                                                </a>
                                                <a href="<?= url('admin/clients.php?edit=' . $client['id']) ?>" 
                                                   class="btn btn-warning" title="Edit Client">
                                                    <i class="bi bi-pencil"></i>
                                                </a>
                                                <form method="POST" action="<?= url('admin/clients.php') ?>" class="d-inline" 
                                                      onsubmit="return confirm('Are you sure you want to deactivate this client?');">
                                                    <?= csrfField() ?>
                                                    <input type="hidden" name="action" value="delete_user">
                                                    <input type="hidden" name="client_id" value="<?= $client['id'] ?>">
                                                    <button type="submit" class="btn btn-danger" title="Deactivate">
                                                        <i class="bi bi-trash"></i>
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <nav aria-label="Page navigation">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>&subscription=<?= $subscriptionFilter ?>">Previous</a>
                                    </li>
                                <?php endif; ?>
                                
                                <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                    <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                        <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>&subscription=<?= $subscriptionFilter ?>"><?= $i ?></a>
                                    </li>
                                <?php endfor; ?>
                                
                                <?php if ($page < $totalPages): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>&subscription=<?= $subscriptionFilter ?>">Next</a>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Create User Modal -->
<div class="modal fade" id="createUserModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Client</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="<?= url('admin/clients.php') ?>">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="create_user">
                    
                    <?php if ($error && !$editingClient): ?>
                        <div class="alert alert-danger">
                            <?= escape($error) ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="email" class="form-label">Email Address <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" id="email" name="email" 
                                   value="<?= escape($formData['email'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="password" class="form-label">Password <span class="text-danger">*</span></label>
                            <input type="password" class="form-control" id="password" name="password" required minlength="8">
                            <small class="form-text text-muted">Minimum 8 characters</small>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="business_name" class="form-label">Business Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="business_name" name="business_name" 
                                   value="<?= escape($formData['business_name'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="contact_name" class="form-label">Contact Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="contact_name" name="contact_name" 
                                   value="<?= escape($formData['contact_name'] ?? '') ?>" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="phone" class="form-label">Phone</label>
                            <input type="tel" class="form-control" id="phone" name="phone" 
                                   value="<?= escape($formData['phone'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="zip_code" class="form-label">Zip Code</label>
                            <input type="text" class="form-control" id="zip_code" name="zip_code" 
                                   value="<?= escape($formData['zip_code'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="address" class="form-label">Address</label>
                        <input type="text" class="form-control" id="address" name="address" 
                               value="<?= escape($formData['address'] ?? '') ?>">
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="city" class="form-label">City</label>
                            <input type="text" class="form-control" id="city" name="city" 
                                   value="<?= escape($formData['city'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="state" class="form-label">State</label>
                            <input type="text" class="form-control" id="state" name="state" 
                                   value="<?= escape($formData['state'] ?? '') ?>">
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Client</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit User Modal -->
<?php if ($editingClient): ?>
<div class="modal fade show" id="editUserModal" tabindex="-1" style="display: block;">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Client: <?= escape($editingClient['business_name']) ?></h5>
                <a href="<?= url('admin/clients.php') ?>" class="btn-close"></a>
            </div>
            <form method="POST" action="<?= url('admin/clients.php') ?>">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="update_user">
                    <input type="hidden" name="client_id" value="<?= $editingClient['id'] ?>">
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_email" class="form-label">Email Address <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" id="edit_email" name="email" 
                                   value="<?= escape($formData['email'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_password" class="form-label">New Password</label>
                            <input type="password" class="form-control" id="edit_password" name="password" minlength="8">
                            <small class="form-text text-muted">Leave empty to keep current password</small>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_business_name" class="form-label">Business Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="edit_business_name" name="business_name" 
                                   value="<?= escape($formData['business_name'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_contact_name" class="form-label">Contact Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="edit_contact_name" name="contact_name" 
                                   value="<?= escape($formData['contact_name'] ?? '') ?>" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_phone" class="form-label">Phone</label>
                            <input type="tel" class="form-control" id="edit_phone" name="phone" 
                                   value="<?= escape($formData['phone'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_status" class="form-label">Status</label>
                            <select class="form-select" id="edit_status" name="status">
                                <option value="active" <?= ($formData['status'] ?? '') === 'active' ? 'selected' : '' ?>>Active</option>
                                <option value="inactive" <?= ($formData['status'] ?? '') === 'inactive' ? 'selected' : '' ?>>Inactive</option>
                                <option value="suspended" <?= ($formData['status'] ?? '') === 'suspended' ? 'selected' : '' ?>>Suspended</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit_address" class="form-label">Address</label>
                        <input type="text" class="form-control" id="edit_address" name="address" 
                               value="<?= escape($formData['address'] ?? '') ?>">
                    </div>
                    
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label for="edit_city" class="form-label">City</label>
                            <input type="text" class="form-control" id="edit_city" name="city" 
                                   value="<?= escape($formData['city'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="edit_state" class="form-label">State</label>
                            <input type="text" class="form-control" id="edit_state" name="state" 
                                   value="<?= escape($formData['state'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="edit_zip_code" class="form-label">Zip Code</label>
                            <input type="text" class="form-control" id="edit_zip_code" name="zip_code" 
                                   value="<?= escape($formData['zip_code'] ?? '') ?>">
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <a href="<?= url('admin/clients.php') ?>" class="btn btn-secondary">Cancel</a>
                    <button type="submit" class="btn btn-primary">Update Client</button>
                </div>
            </form>
        </div>
    </div>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var modal = new bootstrap.Modal(document.getElementById('editUserModal'));
    modal.show();
});
</script>
<?php endif; ?>

<?php if ($error && !$editingClient): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var modal = new bootstrap.Modal(document.getElementById('createUserModal'));
    modal.show();
});
</script>
<?php endif; ?>

<script>
// Bulk actions JavaScript
function toggleAll(checkbox) {
    const checkboxes = document.querySelectorAll('.client-checkbox');
    checkboxes.forEach(cb => cb.checked = checkbox.checked);
    updateBulkButton();
}

function selectAll() {
    const checkboxes = document.querySelectorAll('.client-checkbox');
    checkboxes.forEach(cb => cb.checked = true);
    document.getElementById('selectAllCheckbox').checked = true;
    updateBulkButton();
}

function selectNone() {
    const checkboxes = document.querySelectorAll('.client-checkbox');
    checkboxes.forEach(cb => cb.checked = false);
    document.getElementById('selectAllCheckbox').checked = false;
    updateBulkButton();
}

function updateBulkButton() {
    const checked = document.querySelectorAll('.client-checkbox:checked').length;
    const bulkBtn = document.getElementById('bulkActionBtn');
    const bulkSelect = document.getElementById('bulk_action');
    
    if (checked > 0 && bulkSelect && bulkSelect.value) {
        bulkBtn.disabled = false;
    } else {
        if (bulkBtn) bulkBtn.disabled = true;
    }
}

// Update button state when dropdown changes
document.addEventListener('DOMContentLoaded', function() {
    const bulkSelect = document.getElementById('bulk_action');
    if (bulkSelect) {
        bulkSelect.addEventListener('change', updateBulkButton);
    }
    
    // Confirm bulk actions
    const bulkForm = document.getElementById('bulkActionForm');
    if (bulkForm) {
        bulkForm.addEventListener('submit', function(e) {
            const checked = document.querySelectorAll('.client-checkbox:checked').length;
            const action = document.getElementById('bulk_action').value;
            if (checked === 0) {
                e.preventDefault();
                alert('Please select at least one client');
                return false;
            }
            if (!confirm(`Are you sure you want to ${action} ${checked} client(s)?`)) {
                e.preventDefault();
                return false;
            }
        });
    }
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>


