<?php
/**
 * Admin - Affiliate Partner Management
 * Manage affiliate partners, commissions, and payouts
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/PartnerAuth.php';

requireAdmin();
$pageTitle = 'Affiliate Management';

$db = getDB();
$error = '';
$success = '';

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Search and Filter
$search = trim($_GET['search'] ?? '');
$statusFilter = $_GET['status'] ?? 'all';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requireCSRFToken();
    
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create_partner') {
        $partnerName = trim($_POST['partner_name'] ?? '');
        $contactEmail = trim($_POST['contact_email'] ?? '');
        $contactPhone = trim($_POST['contact_phone'] ?? '');
        $companyName = trim($_POST['company_name'] ?? '');
        $website = trim($_POST['website'] ?? '');
        $commissionRate = floatval($_POST['commission_rate'] ?? 10.0);
        $commissionType = $_POST['commission_type'] ?? 'percentage';
        $paymentMethod = $_POST['payment_method'] ?? 'paypal';
        $paymentEmail = trim($_POST['payment_email'] ?? '');
        
        if (empty($partnerName) || empty($contactEmail)) {
            $error = 'Partner name and email are required';
        } elseif (!filter_var($contactEmail, FILTER_VALIDATE_EMAIL)) {
            $error = 'Valid email address is required';
        } else {
            try {
                $auth = new PartnerAuth();
                $partnerCode = $auth->generatePartnerCode($partnerName);
                
                $stmt = $db->prepare("
                    INSERT INTO affiliate_partners 
                    (partner_code, partner_name, contact_email, contact_phone, company_name, website,
                     commission_rate, commission_type, payment_method, payment_email, created_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $partnerCode, $partnerName, $contactEmail, $contactPhone, $companyName, $website,
                    $commissionRate, $commissionType, $paymentMethod, $paymentEmail, $_SESSION['user_id']
                ]);
                
                $success = "Partner created successfully! Partner Code: <strong>{$partnerCode}</strong>";
            } catch (PDOException $e) {
                $error = 'Error creating partner: ' . $e->getMessage();
            }
        }
    }
}

// Build query with filters
$whereClauses = [];
$params = [];

if (!empty($search)) {
    $whereClauses[] = "(partner_name LIKE ? OR contact_email LIKE ? OR company_name LIKE ? OR partner_code LIKE ?)";
    $searchTerm = '%' . $search . '%';
    $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
}

if ($statusFilter !== 'all') {
    $whereClauses[] = "status = ?";
    $params[] = $statusFilter;
}

$whereClause = !empty($whereClauses) ? 'WHERE ' . implode(' AND ', $whereClauses) : '';

// Get total count
$countQuery = "SELECT COUNT(*) FROM affiliate_partners $whereClause";
$countStmt = $db->prepare($countQuery);
$countStmt->execute($params);
$totalPartners = $countStmt->fetchColumn();
$totalPages = ceil($totalPartners / $perPage);

// Get partners
$query = "SELECT * FROM affiliate_partners $whereClause ORDER BY created_at DESC LIMIT ? OFFSET ?";
$stmt = $db->prepare($query);
$params[] = $perPage;
$params[] = $offset;
$stmt->execute($params);
$partners = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-people"></i> Affiliate Partners</h1>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createPartnerModal">
                <i class="bi bi-plus-circle"></i> Add New Partner
            </button>
        </div>
    </div>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= escape($error) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Search and Filter -->
<div class="row mb-3">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-6">
                        <label for="search" class="form-label">Search</label>
                        <input type="text" class="form-control" id="search" name="search" 
                               value="<?= escape($search) ?>" placeholder="Partner name, email, company...">
                    </div>
                    <div class="col-md-4">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="all" <?= $statusFilter === 'all' ? 'selected' : '' ?>>All Statuses</option>
                            <option value="active" <?= $statusFilter === 'active' ? 'selected' : '' ?>>Active</option>
                            <option value="inactive" <?= $statusFilter === 'inactive' ? 'selected' : '' ?>>Inactive</option>
                            <option value="suspended" <?= $statusFilter === 'suspended' ? 'selected' : '' ?>>Suspended</option>
                        </select>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Partners Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Partners (<?= number_format($totalPartners) ?>)</h5>
            </div>
            <div class="card-body">
                <?php if (empty($partners)): ?>
                    <p class="text-muted mb-0">No partners found. Click "Add New Partner" to create one.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Partner Code</th>
                                    <th>Partner Name</th>
                                    <th>Contact</th>
                                    <th>Commission</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($partners as $partner): ?>
                                    <tr>
                                        <td>
                                            <code><?= escape($partner['partner_code']) ?></code>
                                            <br><small class="text-muted">Referral: <?= url('refer.php?ref=' . $partner['partner_code']) ?></small>
                                        </td>
                                        <td>
                                            <strong><?= escape($partner['partner_name']) ?></strong>
                                            <?php if ($partner['company_name']): ?>
                                                <br><small class="text-muted"><?= escape($partner['company_name']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?= escape($partner['contact_email']) ?>
                                            <?php if ($partner['contact_phone']): ?>
                                                <br><small class="text-muted"><?= escape($partner['contact_phone']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?= $partner['commission_rate'] ?>% 
                                            <small class="text-muted">(<?= ucfirst($partner['commission_type']) ?>)</small>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?= $partner['status'] === 'active' ? 'success' : ($partner['status'] === 'suspended' ? 'danger' : 'secondary') ?>">
                                                <?= escape($partner['status']) ?>
                                            </span>
                                        </td>
                                        <td><?= formatDate($partner['created_at']) ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="<?= url('admin/partner_stats.php?id=' . $partner['id']) ?>" 
                                                   class="btn btn-info" title="View Stats">
                                                    <i class="bi bi-graph-up"></i>
                                                </a>
                                                <a href="<?= url('admin/partner_edit.php?id=' . $partner['id']) ?>" 
                                                   class="btn btn-warning" title="Edit">
                                                    <i class="bi bi-pencil"></i>
                                                </a>
                                                <button class="btn btn-danger" title="Suspend" 
                                                        onclick="suspendPartner(<?= $partner['id'] ?>)">
                                                    <i class="bi bi-pause"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <nav aria-label="Page navigation">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>">Previous</a>
                                    </li>
                                <?php endif; ?>
                                
                                <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                    <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                        <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>"><?= $i ?></a>
                                    </li>
                                <?php endfor; ?>
                                
                                <?php if ($page < $totalPages): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= $statusFilter ?>">Next</a>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Create Partner Modal -->
<div class="modal fade" id="createPartnerModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Affiliate Partner</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="create_partner">
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="partner_name" class="form-label">Partner Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="partner_name" name="partner_name" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="contact_email" class="form-label">Contact Email <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" id="contact_email" name="contact_email" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="contact_phone" class="form-label">Contact Phone</label>
                            <input type="tel" class="form-control" id="contact_phone" name="contact_phone">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="company_name" class="form-label">Company Name</label>
                            <input type="text" class="form-control" id="company_name" name="company_name">
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="website" class="form-label">Website</label>
                            <input type="url" class="form-control" id="website" name="website" placeholder="https://...">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="commission_rate" class="form-label">Commission Rate (%)</label>
                            <input type="number" class="form-control" id="commission_rate" name="commission_rate" 
                                   value="10" min="0" max="100" step="0.01">
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="commission_type" class="form-label">Commission Type</label>
                            <select class="form-select" id="commission_type" name="commission_type">
                                <option value="percentage">Percentage</option>
                                <option value="fixed">Fixed Amount</option>
                                <option value="tiered">Tiered</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="payment_method" class="form-label">Payment Method</label>
                            <select class="form-select" id="payment_method" name="payment_method">
                                <option value="paypal">PayPal</option>
                                <option value="bank_transfer">Bank Transfer</option>
                                <option value="check">Check</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="payment_email" class="form-label">Payment Email/Account</label>
                        <input type="text" class="form-control" id="payment_email" name="payment_email" 
                               placeholder="PayPal email or account details">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Partner</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function suspendPartner(partnerId) {
    if (confirm('Are you sure you want to suspend this partner?')) {
        // Implement suspend functionality
        alert('Suspend functionality not yet implemented');
    }
}
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
