<?php
/**
 * Admin - Activity Log Viewer
 * Displays admin activity logs for audit trail
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/activity_log.php';

requireAdmin();
$pageTitle = 'Activity Log';

$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 50;
$offset = ($page - 1) * $perPage;

// Filters
$actionFilter = $_GET['action'] ?? 'all';
$entityFilter = $_GET['entity'] ?? 'all';
$adminFilter = $_GET['admin'] ?? 'all';

$filters = [];
if ($actionFilter !== 'all') {
    $filters['action_type'] = $actionFilter;
}
if ($entityFilter !== 'all') {
    $filters['entity_type'] = $entityFilter;
}
if ($adminFilter !== 'all' && intval($adminFilter) > 0) {
    $filters['admin_id'] = intval($adminFilter);
}

// Get logs
$logs = getActivityLogs($perPage, $offset, $filters);
$totalLogs = getActivityLogCount($filters);
$totalPages = ceil($totalLogs / $perPage);

// Get admin list for filter
$db = getDB();
$stmt = $db->query("SELECT id, email, business_name FROM users WHERE role = 'admin' ORDER BY email");
$admins = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <h1>Activity Log</h1>
        <p class="text-muted">Audit trail of all admin actions</p>
    </div>
</div>

<!-- Filters -->
<div class="row mb-3">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <form method="GET" action="/admin/activity_log.php" class="row g-3">
                    <div class="col-md-3">
                        <label for="action" class="form-label">Action Type</label>
                        <select class="form-select" id="action" name="action">
                            <option value="all" <?= $actionFilter === 'all' ? 'selected' : '' ?>>All Actions</option>
                            <option value="create" <?= $actionFilter === 'create' ? 'selected' : '' ?>>Create</option>
                            <option value="update" <?= $actionFilter === 'update' ? 'selected' : '' ?>>Update</option>
                            <option value="delete" <?= $actionFilter === 'delete' ? 'selected' : '' ?>>Delete</option>
                            <option value="activate" <?= $actionFilter === 'activate' ? 'selected' : '' ?>>Activate</option>
                            <option value="deactivate" <?= $actionFilter === 'deactivate' ? 'selected' : '' ?>>Deactivate</option>
                            <option value="suspend" <?= $actionFilter === 'suspend' ? 'selected' : '' ?>>Suspend</option>
                            <option value="bulk_action" <?= $actionFilter === 'bulk_action' ? 'selected' : '' ?>>Bulk Action</option>
                            <option value="export" <?= $actionFilter === 'export' ? 'selected' : '' ?>>Export</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="entity" class="form-label">Entity Type</label>
                        <select class="form-select" id="entity" name="entity">
                            <option value="all" <?= $entityFilter === 'all' ? 'selected' : '' ?>>All Entities</option>
                            <option value="user" <?= $entityFilter === 'user' ? 'selected' : '' ?>>User</option>
                            <option value="subscription" <?= $entityFilter === 'subscription' ? 'selected' : '' ?>>Subscription</option>
                            <option value="campaign" <?= $entityFilter === 'campaign' ? 'selected' : '' ?>>Campaign</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="admin" class="form-label">Admin</label>
                        <select class="form-select" id="admin" name="admin">
                            <option value="all" <?= $adminFilter === 'all' ? 'selected' : '' ?>>All Admins</option>
                            <?php foreach ($admins as $admin): ?>
                                <option value="<?= $admin['id'] ?>" <?= $adminFilter == $admin['id'] ? 'selected' : '' ?>>
                                    <?= escape($admin['email']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                    </div>
                </form>
            </div>
        </div>
                    </div>
</div>

<!-- Activity Log Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Activity Log (<?= number_format($totalLogs) ?> entries)</h5>
            </div>
            <div class="card-body">
                <?php if (empty($logs)): ?>
                    <p class="text-muted mb-0">No activity logs found.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-sm">
                            <thead>
                                <tr>
                                    <th>Timestamp</th>
                                    <th>Admin</th>
                                    <th>Action</th>
                                    <th>Entity</th>
                                    <th>Description</th>
                                    <th>Affected</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($logs as $log): ?>
                                    <tr>
                                        <td><?= formatDateTime($log['created_at'], 'Y-m-d H:i:s') ?></td>
                                        <td><?= escape($log['admin_email']) ?></td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                in_array($log['action_type'], ['create', 'activate']) ? 'success' : 
                                                (in_array($log['action_type'], ['update']) ? 'warning' : 
                                                (in_array($log['action_type'], ['delete', 'deactivate', 'suspend']) ? 'danger' : 'info'))
                                            ?>">
                                                <?= escape(ucfirst(str_replace('_', ' ', $log['action_type']))) ?>
                                            </span>
                                        </td>
                                        <td><?= escape(ucfirst($log['entity_type'])) ?></td>
                                        <td><?= escape($log['description'] ?? 'N/A') ?></td>
                                        <td>
                                            <?php if ($log['affected_count'] > 1): ?>
                                                <span class="badge bg-secondary"><?= $log['affected_count'] ?> items</span>
                                            <?php elseif ($log['entity_id']): ?>
                                                ID: <?= $log['entity_id'] ?>
                                            <?php else: ?>
                                                -
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <nav aria-label="Page navigation">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page - 1 ?>&action=<?= $actionFilter ?>&entity=<?= $entityFilter ?>&admin=<?= $adminFilter ?>">Previous</a>
                                    </li>
                                <?php endif; ?>
                                
                                <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                    <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                        <a class="page-link" href="?page=<?= $i ?>&action=<?= $actionFilter ?>&entity=<?= $entityFilter ?>&admin=<?= $adminFilter ?>"><?= $i ?></a>
                                    </li>
                                <?php endfor; ?>
                                
                                <?php if ($page < $totalPages): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page + 1 ?>&action=<?= $actionFilter ?>&entity=<?= $entityFilter ?>&admin=<?= $adminFilter ?>">Next</a>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

