<?php
/**
 * Ad Generator - Website Creator for Affiliate Marketing
 * Generates complete websites for affiliate marketing campaigns
 * Saves generated sites to affiliates-pages folder
 */

session_start();

require_once 'ga4-config.php';

if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    header('Location: index.php');
    exit;
}

// Create affiliates-pages directory if it doesn't exist
$affiliatesDir = __DIR__ . '/../affiliates-pages';
if (!is_dir($affiliatesDir)) {
    mkdir($affiliatesDir, 0755, true);
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <title>Ad Generator - Website Creator - Marketing Platform</title>
    
    <?php 
    echo GA4Config::getHeadScript();
    echo GA4Config::trackPageView('ad_generator');
    ?>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="universal-data-handler.js"></script>
    <script async src="https://www.googletagmanager.com/gtag/js?id=G-SLMXCVWD2T"></script>
    <script>
      window.dataLayer = window.dataLayer || [];
      function gtag(){dataLayer.push(arguments);}
      gtag('js', new Date());
      gtag('config', 'G-SLMXCVWD2T');
    </script>
    
    <style>
        .form-input, .form-select, .form-textarea {
            background-color: #374151; 
            border-color: #4B5563; 
            color: #F9FAFB;
            transition: all 0.3s ease;
            border-width: 2px;
        }
        .form-input:focus, .form-select:focus, .form-textarea:focus {
            --tw-ring-color: #4F46E5; 
            border-color: #4F46E5;
            background-color: #4B5563;
        }
        
        .btn-awesome {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            position: relative;
            overflow: hidden;
            transition: all 0.3s ease;
        }
        .btn-awesome:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.4);
        }
        
        .gradient-text {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 50%, #f093fb 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            color: #667eea;
        }
        
        #website-preview {
            width: 100%; 
            height: 100%; 
            border: none; 
            background-color: #fff;
            border-radius: 8px;
        }
    </style>
</head>

<body class="bg-gray-900 text-white antialiased">
    <div class="container mx-auto p-4 sm:p-6 md:p-8">
        <header class="bg-slate-800 border-b border-slate-700 px-6 py-4 mb-8 rounded-lg">
            <div class="flex items-center justify-between">
                <div class="flex items-center gap-3">
                    <button onclick="goBack()" class="flex items-center gap-2 bg-slate-700 hover:bg-slate-600 text-white px-4 py-2 rounded-lg transition-colors">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path></svg>
                        <span class="font-medium">Back</span>
                    </button>
                </div>
                <div class="flex items-center gap-3">
                    <a href="index.php" class="text-sm text-gray-400 hover:text-white transition-colors">
                        ← Return to Dashboard
                    </a>
                </div>
            </div>
            <div class="text-center mt-6">
                <h1 class="text-4xl md:text-5xl font-bold text-cyan-400">
                    Ad Generator - Website Creator
                </h1>
                <p class="text-gray-400 mt-2">Generate complete websites for affiliate marketing campaigns</p>
            </div>
        </header>

        <div id="message-box" class="mt-4 mb-6"></div>

        <div class="grid grid-cols-1 xl:grid-cols-3 gap-8">
            <!-- Left Column: Campaign Details -->
            <div class="bg-gray-800 p-6 rounded-2xl shadow-lg border border-gray-700 flex flex-col space-y-6">
                <div>
                    <h2 class="text-2xl font-bold mb-4 gradient-text">🎯 Campaign Details</h2>
                    <div class="space-y-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div class="form-group">
                                <label for="campaign-name" class="block text-sm font-medium text-gray-300 mb-2">✨ Campaign Name</label>
                                <input type="text" id="campaign-name" class="w-full form-input rounded-lg p-3" placeholder="e.g., 'Summer Sale Campaign'">
                            </div>
                            <div class="form-group">
                                <label for="business-type" class="block text-sm font-medium text-gray-300 mb-2">🏢 Business Type</label>
                                <select id="business-type" class="w-full form-select rounded-lg p-3">
                                    <option value="">Select Business Type</option>
                                    <option value="restaurant">Restaurant</option>
                                    <option value="retail">Retail Store</option>
                                    <option value="service">Service Business</option>
                                    <option value="consulting">Consulting</option>
                                    <option value="ecommerce">E-commerce</option>
                                    <option value="healthcare">Healthcare</option>
                                    <option value="fitness">Fitness</option>
                                    <option value="beauty">Beauty & Wellness</option>
                                    <option value="real_estate">Real Estate</option>
                                    <option value="automotive">Automotive</option>
                                    <option value="education">Education</option>
                                    <option value="technology">Technology</option>
                                    <option value="finance">Finance</option>
                                    <option value="other">Other</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div class="form-group">
                                <label for="target-audience" class="block text-sm font-medium text-gray-300 mb-2">🎯 Target Audience</label>
                                <input type="text" id="target-audience" class="w-full form-input rounded-lg p-3" placeholder="e.g., Young professionals, Families, Seniors">
                            </div>
                            <div class="form-group">
                                <label for="campaign-goal" class="block text-sm font-medium text-gray-300 mb-2">🎯 Campaign Goal</label>
                                <select id="campaign-goal" class="w-full form-select rounded-lg p-3">
                                    <option value="lead_generation">Lead Generation</option>
                                    <option value="sales">Direct Sales</option>
                                    <option value="brand_awareness">Brand Awareness</option>
                                    <option value="newsletter_signup">Newsletter Signup</option>
                                    <option value="appointment_booking">Appointment Booking</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="key-features" class="block text-sm font-medium text-gray-300 mb-2">🚀 Website Features</label>
                            <textarea id="key-features" rows="4" class="w-full form-textarea rounded-lg p-3" placeholder="e.g., 'Hero section with CTA, Product showcase, Contact form, Testimonials, Pricing table'"></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="campaign-message" class="block text-sm font-medium text-gray-300 mb-2">💬 Campaign Message</label>
                            <textarea id="campaign-message" rows="3" class="w-full form-textarea rounded-lg p-3" placeholder="Key message you want to convey to visitors..."></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="special-requirements" class="block text-sm font-medium text-gray-300 mb-2">📝 Campaign Notes</label>
                            <textarea id="special-requirements" rows="3" class="w-full form-textarea rounded-lg p-3" placeholder="Any specific messaging, offers, or campaign requirements..."></textarea>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Middle Column: Content Preferences -->
            <div class="bg-gray-800 p-6 rounded-2xl shadow-lg border border-gray-700 flex flex-col space-y-6">
                <div>
                    <h2 class="text-2xl font-bold mb-4 gradient-text">💡 Content Preferences</h2>
                    <p class="text-gray-400 text-sm mb-4">Customize the content generation (company details will be kept separate)</p>
                    
                    <div class="space-y-4">
                        <div class="form-group">
                            <label for="brand-voice" class="block text-sm font-medium text-gray-300 mb-2">🎭 Brand Voice & Tone</label>
                            <select id="brand-voice" class="w-full form-select rounded-lg p-3">
                                <option value="professional">Professional & Corporate</option>
                                <option value="friendly">Friendly & Approachable</option>
                                <option value="creative">Creative & Artistic</option>
                                <option value="modern">Modern & Tech-Savvy</option>
                                <option value="luxury">Luxury & Premium</option>
                                <option value="casual">Casual & Relaxed</option>
                                <option value="energetic">Energetic & Dynamic</option>
                                <option value="trustworthy">Trustworthy & Reliable</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="color-scheme" class="block text-sm font-medium text-gray-300 mb-2">🎨 Color Scheme</label>
                            <select id="color-scheme" class="w-full form-select rounded-lg p-3">
                                <option value="blue">Professional Blue</option>
                                <option value="green">Nature Green</option>
                                <option value="purple">Creative Purple</option>
                                <option value="orange">Energetic Orange</option>
                                <option value="red">Bold Red</option>
                                <option value="teal">Modern Teal</option>
                                <option value="pink">Friendly Pink</option>
                                <option value="dark">Dark Theme</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="layout-style" class="block text-sm font-medium text-gray-300 mb-2">📐 Layout Style</label>
                            <select id="layout-style" class="w-full form-select rounded-lg p-3">
                                <option value="modern">Modern & Clean</option>
                                <option value="minimalist">Minimalist</option>
                                <option value="bold">Bold & Dynamic</option>
                                <option value="elegant">Elegant & Sophisticated</option>
                                <option value="playful">Playful & Fun</option>
                                <option value="corporate">Corporate & Professional</option>
                            </select>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Right Column: Preview & Actions -->
            <div class="bg-gray-800 p-6 rounded-2xl shadow-lg border border-gray-700 flex flex-col space-y-6">
                <div>
                    <h2 class="text-2xl font-bold mb-4 gradient-text">👁️ Preview & Actions</h2>
                    
                    <div class="space-y-4">
                        <button id="btn-generate-code" class="w-full btn-awesome text-white font-bold py-3 px-6 rounded-lg transition-all duration-300 hover:scale-105">
                            <i class="bi bi-megaphone"></i> Generate Ad Website
                        </button>
                        
                        <button id="btn-download-code" class="w-full bg-green-600 hover:bg-green-700 text-white font-bold py-3 px-6 rounded-lg transition-all duration-300 hover:scale-105" disabled>
                            <i class="bi bi-download"></i> Download Website
                        </button>
                        
                        <div class="bg-gray-700/50 p-4 rounded-lg">
                            <h3 class="text-lg font-semibold text-gray-200 mb-2">📁 Save Location</h3>
                            <p class="text-sm text-gray-400">Generated websites will be saved to:</p>
                            <code class="text-xs text-cyan-400">affiliates-pages/</code>
                        </div>
                    </div>
                </div>
                
                <div class="flex-grow">
                    <h3 class="text-lg font-semibold text-gray-200 mb-3">🔍 Live Preview</h3>
                    <div class="bg-white rounded-lg overflow-hidden" style="height: 400px;">
                        <iframe id="website-preview" title="Website Preview" style="width: 100%; height: 100%; border: none;"></iframe>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        function goBack() {
            if (window.history.length > 1) {
                window.history.back();
            } else {
                window.location.href = 'index.php';
            }
        }

        document.addEventListener('DOMContentLoaded', () => {
            const campaignNameEl = document.getElementById('campaign-name');
            const businessTypeEl = document.getElementById('business-type');
            const keyFeaturesEl = document.getElementById('key-features');
            const generateCodeBtn = document.getElementById('btn-generate-code');
            const downloadCodeBtn = document.getElementById('btn-download-code');
            const previewFrame = document.getElementById('website-preview');
            const messageBox = document.getElementById('message-box');

            const showMessage = (message, type = 'info') => {
                const colorClass = type === 'error' ? 'bg-red-900/50 text-red-300 border-red-700' : 'bg-blue-900/50 text-blue-300 border-blue-700';
                messageBox.innerHTML = `<div class="p-4 rounded-lg border ${colorClass}">${message}</div>`;
            };

            const setLoadingState = (button, isLoading, text = '') => {
                if (isLoading) {
                    button.disabled = true;
                    button.innerHTML = `<div class="spinner inline-block mr-2"></div> ${text}`;
                } else {
                    button.disabled = false;
                    button.innerHTML = '<i class="bi bi-megaphone"></i> Generate Ad Website';
                }
            };

            const callGateway = async (payload) => {
                try {
                    const response = await fetch('../api/gateway.php?endpoint=chat', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        credentials: 'include',
                        body: JSON.stringify(payload)
                    });
                    
                    if (!response.ok) {
                        if (response.status === 401) {
                            window.location.reload();
                            return null;
                        }
                        throw new Error(`API Error: ${response.status}`);
                    }
                    
                    const responseText = await response.text();
                    if (!responseText.trim()) {
                        throw new Error('Empty response from server');
                    }
                    
                    return JSON.parse(responseText);
                } catch (error) {
                    console.error('Gateway call failed:', error);
                    showMessage(error.message, 'error');
                    return null;
                }
            };

            generateCodeBtn.addEventListener('click', async () => {
                const campaignName = campaignNameEl.value.trim();
                const businessType = businessTypeEl.value.trim();
                const keyFeatures = keyFeaturesEl.value.trim();
                const targetAudience = document.getElementById('target-audience').value.trim();
                const campaignGoal = document.getElementById('campaign-goal').value;
                const campaignMessage = document.getElementById('campaign-message').value.trim();
                const brandVoice = document.getElementById('brand-voice').value;
                const colorScheme = document.getElementById('color-scheme').value;
                const layoutStyle = document.getElementById('layout-style').value;

                if (!campaignName || !businessType || !keyFeatures) {
                    showMessage('Please fill out all required fields to generate the website.', 'error');
                    return;
                }

                setLoadingState(generateCodeBtn, true, 'Generating Ad Website...');
                downloadCodeBtn.disabled = true;

                try {
                    // Generate content without company details
                    const contentPrompt = `You are a Content Generator Agent for affiliate marketing campaigns. Create compelling, professional content for a marketing website.

Campaign Details:
- Campaign Name: "${campaignName}"
- Business Type: "${businessType}"
- Target Audience: "${targetAudience || 'General audience'}"
- Campaign Goal: "${campaignGoal}"
- Key Features: "${keyFeatures}"
- Campaign Message: "${campaignMessage || 'Discover our amazing solutions'}"
- Brand Voice: "${brandVoice}"

IMPORTANT: Do NOT include any specific company names, contact details, addresses, or personal information. Generate generic, professional content that can be used for any business in this industry.

Generate the following content in JSON format:
{
  "heroTitle": "Compelling headline for ${businessType} campaign",
  "heroSubtitle": "Engaging subtitle that appeals to ${targetAudience || 'target audience'}",
  "heroCallToAction": "Call to action button text",
  "aboutTitle": "Why Choose Our ${businessType} Solutions?",
  "aboutContent": "Professional description of ${businessType} benefits and value proposition",
  "servicesTitle": "Our ${businessType} Services",
  "services": [
    {"title": "Service 1", "description": "Professional service description"},
    {"title": "Service 2", "description": "Professional service description"},
    {"title": "Service 3", "description": "Professional service description"}
  ],
  "contactTitle": "Ready to Get Started?",
  "contactDescription": "Professional contact invitation text",
  "footerText": "Professional footer tagline",
  "campaignMessage": "${campaignMessage || 'Transform your business with our professional solutions'}"
}

OUTPUT: Only valid JSON, no explanations or markdown.`;

                    const contentPayload = {
                        model: 'together/meta-llama/Llama-3.3-70B-Instruct-Turbo-Free',
                        messages: [{ role: 'user', content: contentPrompt }],
                        max_tokens: 2048
                    };

                    const contentResult = await callGateway(contentPayload);
                    
                    if (!contentResult || !contentResult.choices || !contentResult.choices[0].message) {
                        throw new Error('Content Generator Agent failed to respond');
                    }

                    let contentData;
                    try {
                        const contentText = contentResult.choices[0].message.content.trim();
                        const cleanContent = contentText.replace(/^```json\n/, '').replace(/\n```$/, '');
                        contentData = JSON.parse(cleanContent);
                    } catch (e) {
                        throw new Error('Content Generator Agent returned invalid JSON');
                    }

                    // Generate website structure
                    const structurePrompt = `You are a Structure Generator Agent. Create a complete, stunning marketing website with all content filled in.

CONTENT TO USE:
- Hero Title: "${contentData.heroTitle}"
- Hero Subtitle: "${contentData.heroSubtitle}"
- Hero Call to Action: "${contentData.heroCallToAction}"
- About Title: "${contentData.aboutTitle}"
- About Content: "${contentData.aboutContent}"
- Services Title: "${contentData.servicesTitle}"
- Services: ${contentData.services.map(s => `"${s.title}: ${s.description}"`).join(', ')}
- Contact Title: "${contentData.contactTitle}"
- Contact Description: "${contentData.contactDescription}"
- Footer Text: "${contentData.footerText}"
- Campaign Message: "${contentData.campaignMessage}"

DESIGN SPECIFICATIONS:
- Use Tailwind CSS CDN for styling
- Color Scheme: ${colorScheme} (implement this color scheme throughout)
- Layout Style: ${layoutStyle} (follow this layout approach)
- Brand Voice: Ensure all text matches the ${brandVoice} tone
- Responsive Design: Ensure perfect mobile, tablet, and desktop layouts
- Modern UI: Include hover effects, transitions, animations, and micro-interactions
- Typography: Use modern font combinations with proper hierarchy

REQUIRED SECTIONS:
- Hero section with compelling headline and CTA
- About section explaining value proposition
- Services/Features section showcasing offerings
- Contact section with form (generic, no specific contact details)
- Footer with professional tagline

INTERACTIVE ELEMENTS:
- Smooth scrolling navigation
- Hover effects on buttons and cards
- Contact form with validation
- Responsive mobile menu
- Call-to-action buttons throughout

OUTPUT: Complete HTML document starting with <!DOCTYPE html> and ending with </html>. Use ONLY the actual content provided above. NO placeholders, NO thinking, NO explanations.`;

                    const structurePayload = {
                        model: 'together/deepseek-ai/DeepSeek-R1-Distill-Llama-70B-free',
                        messages: [{ role: 'user', content: structurePrompt }],
                        max_tokens: 4096
                    };

                    const structureResult = await callGateway(structurePayload);
                    setLoadingState(generateCodeBtn, false);

                    if (structureResult && structureResult.choices && structureResult.choices[0].message) {
                        let generatedCode = structureResult.choices[0].message.content.trim();
                        
                        // Clean up the generated code
                        generatedCode = generatedCode.replace(/^```html\n/, '').replace(/\n```$/, '');
                        generatedCode = generatedCode.replace(/^```\n/, '').replace(/\n```$/, '');
                        
                        // Extract only the HTML document
                        const htmlStart = generatedCode.indexOf('<!DOCTYPE html>');
                        const htmlEnd = generatedCode.lastIndexOf('</html>') + 7;
                        
                        if (htmlStart !== -1 && htmlEnd !== -1) {
                            generatedCode = generatedCode.substring(htmlStart, htmlEnd);
                        }
                        
                        generatedCode = generatedCode.replace(/^[^<]*<!DOCTYPE/, '<!DOCTYPE');
                        generatedCode = generatedCode.replace(/<\/html>[^<]*$/, '</html>');
                        
                        if (!generatedCode.startsWith('<!DOCTYPE html>')) {
                            throw new Error('Structure Generator did not produce valid HTML document');
                        }
                        
                        // Save to affiliates-pages folder
                        const filename = 'ad_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9) + '.html';
                        const filepath = '<?= $affiliatesDir ?>/' + filename;
                        
                        // Create a form to submit the generated code to a PHP handler
                        const form = document.createElement('form');
                        form.method = 'POST';
                        form.action = 'save-website.php';
                        form.style.display = 'none';
                        
                        const codeInput = document.createElement('input');
                        codeInput.type = 'hidden';
                        codeInput.name = 'website_code';
                        codeInput.value = generatedCode;
                        
                        const filenameInput = document.createElement('input');
                        filenameInput.type = 'hidden';
                        filenameInput.name = 'filename';
                        filenameInput.value = filename;
                        
                        form.appendChild(codeInput);
                        form.appendChild(filenameInput);
                        document.body.appendChild(form);
                        form.submit();
                        
                        // Show preview
                        previewFrame.srcdoc = generatedCode;
                        downloadCodeBtn.disabled = false;
                        showMessage('🎉 Ad website generated successfully! Saved to affiliates-pages folder.', 'info');
                        
                    } else {
                        throw new Error('Structure Generator Agent failed to create website');
                    }

                } catch (error) {
                    setLoadingState(generateCodeBtn, false);
                    showMessage(`❌ Generation failed: ${error.message}`, 'error');
                    previewFrame.srcdoc = `<div style="display:flex;justify-content:center;align-items:center;height:100%;font-family:sans-serif;color:red;">Generation Failed: ${error.message}</div>`;
                }
            });

            downloadCodeBtn.addEventListener('click', () => {
                const codeToDownload = previewFrame.srcdoc;
                if (!codeToDownload) {
                    showMessage('There is no website to download.', 'error');
                    return;
                }

                const blob = new Blob([codeToDownload], { type: 'text/html' });
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = 'ad-website.html';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                URL.revokeObjectURL(url);
            });
        });
    </script>
</body>
</html>
