<?php
/**
 * AcceleratedLeadz - Public Lead Capture Form
 * Displays lead capture forms at ai-itinc.com/acceleratedleadz?id=123
 * No authentication required - public facing
 */

require_once __DIR__ . '/includes/base_path.php';
require_once __DIR__ . '/includes/database.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/AffiliateTracker.php';

// Start session for affiliate tracking
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Get form ID, campaign ID, affiliate code from URL
$formId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$campaignId = isset($_GET['campaign']) ? (int)$_GET['campaign'] : 0;
$userId = isset($_GET['user']) ? (int)$_GET['user'] : 0;
$affiliateCode = isset($_GET['ref']) ? trim($_GET['ref']) : '';

// Track affiliate click if code provided
$affiliateTracked = false;
if (!empty($affiliateCode)) {
    $tracker = new AffiliateTracker();
    $clickId = $tracker->trackClick($affiliateCode, $_SERVER['HTTP_REFERER'] ?? null, $_SERVER['REQUEST_URI'] ?? null);
    $affiliateTracked = true;
    // Store in session for form submission
    $_SESSION['affiliate_code'] = $affiliateCode;
}

$db = getDB();
$success = false;
$error = '';
$formData = [];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    $firstName = trim($_POST['first_name'] ?? '');
    $lastName = trim($_POST['last_name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $companyName = trim($_POST['company_name'] ?? '');
    $notes = trim($_POST['notes'] ?? '');
    
    // Get user_id from form submission or URL
    $submitUserId = (int)($_POST['user_id'] ?? $userId ?? 0);
    
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address';
    } else {
        try {
            // Add to email_subscribers (CRM)
            $stmt = $db->prepare("
                INSERT INTO email_subscribers (user_id, email, first_name, last_name, status, subscribed_at)
                VALUES (?, ?, ?, ?, 'subscribed', NOW())
                ON DUPLICATE KEY UPDATE
                    first_name = COALESCE(VALUES(first_name), first_name),
                    last_name = COALESCE(VALUES(last_name), last_name),
                    status = 'subscribed',
                    unsubscribed_at = NULL
            ");
            
            $stmt->execute([
                $submitUserId ?: 0,
                $email,
                $firstName ?: null,
                $lastName ?: null
            ]);
            
            // Also add to client_leads if user_id provided or affiliate code exists
            $fullName = trim(($firstName ?? '') . ' ' . ($lastName ?? ''));
            
            // Get affiliate info if affiliate code in session
            $partnerId = null;
            $partnerCode = null;
            $affiliateClickId = null;
            
            if (isset($_SESSION['affiliate_code'])) {
                $affCode = $_SESSION['affiliate_code'];
                $affStmt = $db->prepare("SELECT id, partner_code FROM affiliate_partners WHERE partner_code = ? AND status = 'active'");
                $affStmt->execute([$affCode]);
                $affiliate = $affStmt->fetch();
                
                if ($affiliate) {
                    $partnerId = $affiliate['id'];
                    $partnerCode = $affiliate['partner_code'];
                    
                    // Get latest click for this affiliate
                    $clickStmt = $db->prepare("SELECT id FROM affiliate_clicks WHERE partner_code = ? ORDER BY clicked_at DESC LIMIT 1");
                    $clickStmt->execute([$partnerCode]);
                    $click = $clickStmt->fetch();
                    $affiliateClickId = $click ? $click['id'] : null;
                }
            }
            
            // Create lead record (always create for tracking)
            $leadStmt = $db->prepare("
                INSERT INTO client_leads 
                (user_id, affiliate_click_id, partner_id, partner_code, lead_email, lead_name, lead_phone, company_name, lead_source, status, notes, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'acceleratedleadz', 'new', ?, NOW())
            ");
            
            $leadStmt->execute([
                $submitUserId ?: null,
                $affiliateClickId,
                $partnerId,
                $partnerCode,
                $email,
                $fullName ?: null,
                $phone ?: null,
                $companyName ?: null,
                $notes ?: null
            ]);
            
            $leadId = $db->lastInsertId();
            
            // Process referral if affiliate exists
            if ($partnerId && $leadId) {
                require_once __DIR__ . '/includes/ReferralProcessor.php';
                $processor = new ReferralProcessor();
                // The lead is already created, just ensure it's linked properly
            }
            
            $success = true;
            
            // Optionally add to email queue if campaign_id provided
            if ($campaignId && $submitUserId) {
                $subscriberId = $db->lastInsertId();
                if ($subscriberId) {
                    // Queue a welcome email or campaign email
                    $queueStmt = $db->prepare("
                        INSERT INTO accelerated_leadz_email_queue 
                        (campaign_id, subscriber_id, email_subject, email_content, email_type, scheduled_at, status)
                        VALUES (?, ?, 'Welcome!', '<h1>Thank you for subscribing!</h1><p>We'll be in touch soon.</p>', 'html', NOW(), 'pending')
                    ");
                    $queueStmt->execute([$campaignId, $subscriberId]);
                }
            }
            
        } catch (PDOException $e) {
            error_log("AcceleratedLeadz form submission error: " . $e->getMessage());
            $error = 'There was an error processing your submission. Please try again.';
        }
    }
}

// Get form configuration if form_id provided (for future customization)
if ($formId) {
    // Could fetch form configuration from database
    // For now, use default form
}

// Get user info if user_id provided (for branding)
$userInfo = null;
if ($userId) {
    $stmt = $db->prepare("SELECT business_name, contact_name FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $userInfo = $stmt->fetch();
}

$businessName = $userInfo['business_name'] ?? 'AI-IT Inc';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Get Started - <?= htmlspecialchars($businessName) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            padding: 20px;
        }
        .lead-form-container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            max-width: 600px;
            width: 100%;
            margin: 0 auto;
            overflow: hidden;
        }
        .lead-form-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 30px;
            text-align: center;
        }
        .lead-form-header h1 {
            font-size: 32px;
            font-weight: bold;
            margin-bottom: 10px;
        }
        .lead-form-header p {
            font-size: 16px;
            opacity: 0.9;
            margin: 0;
        }
        .lead-form-body {
            padding: 40px 30px;
        }
        .form-label {
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
        }
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 12px 15px;
            transition: border-color 0.3s ease;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .btn-submit {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            color: white;
            padding: 15px 40px;
            border-radius: 8px;
            font-weight: bold;
            font-size: 18px;
            width: 100%;
            transition: opacity 0.3s ease;
        }
        .btn-submit:hover {
            opacity: 0.9;
            color: white;
        }
        .success-message {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
        }
        .success-message i {
            font-size: 48px;
            color: #28a745;
            margin-bottom: 15px;
        }
    </style>
</head>
<body>
    <div class="lead-form-container">
        <div class="lead-form-header">
            <h1><i class="bi bi-envelope-heart"></i> Get Started Today</h1>
            <p>Join thousands of satisfied customers</p>
        </div>
        
        <div class="lead-form-body">
            <?php if ($success): ?>
                <div class="success-message">
                    <i class="bi bi-check-circle"></i>
                    <h3>Thank You!</h3>
                    <p>We've received your information and will be in touch soon.</p>
                </div>
            <?php else: ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?= htmlspecialchars($error) ?>
                    </div>
                <?php endif; ?>
                
                <form method="POST" id="leadForm">
                    <input type="hidden" name="user_id" value="<?= $userId ?>">
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="first_name" class="form-label">First Name</label>
                            <input type="text" class="form-control" id="first_name" name="first_name" 
                                   value="<?= htmlspecialchars($_POST['first_name'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="last_name" class="form-label">Last Name</label>
                            <input type="text" class="form-control" id="last_name" name="last_name" 
                                   value="<?= htmlspecialchars($_POST['last_name'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="email" class="form-label">
                            Email Address <span class="text-danger">*</span>
                        </label>
                        <input type="email" class="form-control" id="email" name="email" 
                               value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="phone" class="form-label">Phone Number</label>
                        <input type="tel" class="form-control" id="phone" name="phone" 
                               value="<?= htmlspecialchars($_POST['phone'] ?? '') ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="company_name" class="form-label">Company Name</label>
                        <input type="text" class="form-control" id="company_name" name="company_name" 
                               value="<?= htmlspecialchars($_POST['company_name'] ?? '') ?>">
                    </div>
                    
                    <div class="mb-4">
                        <label for="notes" class="form-label">Additional Information</label>
                        <textarea class="form-control" id="notes" name="notes" rows="3" 
                                  placeholder="Tell us about your needs..."><?= htmlspecialchars($_POST['notes'] ?? '') ?></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-submit">
                        <i class="bi bi-send"></i> Submit
                    </button>
                </form>
                
                <div class="text-center mt-4">
                    <small class="text-muted">
                        By submitting this form, you agree to our 
                        <a href="<?= url('privacy-policy.php') ?>" target="_blank">Privacy Policy</a>
                    </small>
                </div>
            <?php endif; ?>
        </div>
        
        <div class="text-center p-3 bg-light">
            <small class="text-muted">
                Powered by <strong>AcceleratedLeadz</strong> - AI-IT Inc
            </small>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Track form view (optional analytics)
        if (typeof gtag !== 'undefined') {
            gtag('event', 'form_view', {
                'event_category': 'AcceleratedLeadz',
                'event_label': '<?= $formId ?>'
            });
        }
    </script>
</body>
</html>

