<?php
/**
 * AcceleratedAdz - Public Ad Display
 * Displays ads at ai-itinc.com/acceleratedadz?id=123
 * No authentication required - public facing
 */

require_once __DIR__ . '/includes/base_path.php';
require_once __DIR__ . '/includes/database.php';
require_once __DIR__ . '/includes/functions.php';

// Handle click redirect
if (isset($_GET['click'])) {
    $clickAdId = (int)$_GET['click'];
    $redirectUrl = $_GET['redirect'] ?? '#';
    $clickSource = $_GET['source'] ?? 'acceleratedadz';
    
    // Track click
    try {
        $db = getDB();
        
        if ($clickSource === 'campaign') {
            // Track click for campaign_ads (could update ad_performance table if needed)
            // For now, just log it
            error_log("Campaign ad clicked: campaign_ad_id={$clickAdId}");
        } else {
            // Track click for acceleratedadz_ads
            $adStmt = $db->prepare("SELECT ad_group_id, campaign_id FROM acceleratedadz_ads WHERE id = ?");
            $adStmt->execute([$clickAdId]);
            $ad = $adStmt->fetch();
            
            if ($ad) {
                $date = date('Y-m-d');
                $stmt = $db->prepare("
                    INSERT INTO acceleratedadz_performance 
                    (ad_id, ad_group_id, campaign_id, date, impressions, clicks, conversions, spend, revenue, 
                     ctr, cpc, cpm, cpa, roas)
                    VALUES (?, ?, ?, ?, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0)
                    ON DUPLICATE KEY UPDATE
                        clicks = clicks + 1,
                        ctr = CASE WHEN impressions > 0 THEN (clicks + 1) / impressions ELSE 0 END,
                        updated_at = NOW()
                ");
                $stmt->execute([$clickAdId, $ad['ad_group_id'], $ad['campaign_id'], $date]);
            }
        }
    } catch (PDOException $e) {
        error_log("AcceleratedAdz click tracking error: " . $e->getMessage());
    }
    
    // Redirect to destination
    header('Location: ' . $redirectUrl);
    exit;
}

// Get ad ID from URL
$adId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$campaignId = isset($_GET['campaign']) ? (int)$_GET['campaign'] : 0;
$campaignAdId = isset($_GET['campaign_ad']) ? (int)$_GET['campaign_ad'] : 0; // For campaign_ads table
$source = isset($_GET['source']) ? $_GET['source'] : 'acceleratedadz'; // 'acceleratedadz' or 'campaign'

if (!$adId && !$campaignId && !$campaignAdId) {
    http_response_code(404);
    die('Ad not found');
}

$db = getDB();
$ad = null;
$adData = null;

// Get ad details - support both acceleratedadz_ads and campaign_ads
if ($campaignAdId || $source === 'campaign') {
    // Get ad from campaign_ads table (used in campaigns.php)
    $stmt = $db->prepare("
        SELECT ca.*, sa.ad_data, sa.id as saved_ad_id, c.id as campaign_id, c.campaign_name, c.user_id
        FROM campaign_ads ca
        LEFT JOIN saved_ads sa ON ca.saved_ad_id = sa.id
        JOIN campaigns c ON ca.campaign_id = c.id
        WHERE ca.id = ? AND ca.status = 'active'
    ");
    $stmt->execute([$campaignAdId]);
    $campaignAd = $stmt->fetch();
    
    if ($campaignAd) {
        // Convert campaign_ads format to acceleratedadz format for display
        $savedAdData = json_decode($campaignAd['ad_data'] ?? '{}', true);
        $ad = [
            'id' => $campaignAd['id'],
            'ad_name' => $campaignAd['ad_name'],
            'headline' => $savedAdData['headline'] ?? '',
            'description' => $savedAdData['description'] ?? '',
            'image_url' => $savedAdData['productImage'] ?? $savedAdData['bgImage'] ?? '',
            'destination_url' => $savedAdData['destinationUrl'] ?? '#',
            'display_url' => parse_url($savedAdData['destinationUrl'] ?? '', PHP_URL_HOST) ?: 'ai-itinc.com',
            'call_to_action' => $savedAdData['cta'] ?? 'Learn More',
            'ad_type' => !empty($savedAdData['productImage']) ? 'image' : 'text',
            'campaign_id' => $campaignAd['campaign_id'],
            'user_id' => $campaignAd['user_id']
        ];
        $adData = $savedAdData;
        $adId = $campaignAd['id'];
    }
} elseif ($adId) {
    // Get ad from acceleratedadz_ads table
    $stmt = $db->prepare("
        SELECT a.*, ag.campaign_id, c.campaign_name, c.user_id
        FROM acceleratedadz_ads a
        JOIN acceleratedadz_ad_groups ag ON a.ad_group_id = ag.id
        JOIN acceleratedadz_campaigns c ON ag.campaign_id = c.id
        WHERE a.id = ? AND a.status = 'active'
    ");
    $stmt->execute([$adId]);
    $ad = $stmt->fetch();
} elseif ($campaignId) {
    // Try acceleratedadz campaign first, then regular campaign
    $stmt = $db->prepare("
        SELECT a.*, ag.campaign_id, c.campaign_name, c.user_id
        FROM acceleratedadz_ads a
        JOIN acceleratedadz_ad_groups ag ON a.ad_group_id = ag.id
        JOIN acceleratedadz_campaigns c ON ag.campaign_id = c.id
        WHERE c.id = ? AND a.status = 'active' AND c.status = 'active'
        ORDER BY RAND()
        LIMIT 1
    ");
    $stmt->execute([$campaignId]);
    $ad = $stmt->fetch();
    
    if (!$ad) {
        // Fallback to regular campaigns table
        $stmt = $db->prepare("
            SELECT ca.*, sa.ad_data, c.id as campaign_id, c.campaign_name, c.user_id
            FROM campaign_ads ca
            LEFT JOIN saved_ads sa ON ca.saved_ad_id = sa.id
            JOIN campaigns c ON ca.campaign_id = c.id
            WHERE c.id = ? AND ca.status = 'active'
            ORDER BY RAND()
            LIMIT 1
        ");
        $stmt->execute([$campaignId]);
        $campaignAd = $stmt->fetch();
        
        if ($campaignAd) {
            $savedAdData = json_decode($campaignAd['ad_data'] ?? '{}', true);
            $ad = [
                'id' => $campaignAd['id'],
                'ad_name' => $campaignAd['ad_name'],
                'headline' => $savedAdData['headline'] ?? '',
                'description' => $savedAdData['description'] ?? '',
                'image_url' => $savedAdData['productImage'] ?? $savedAdData['bgImage'] ?? '',
                'destination_url' => $savedAdData['destinationUrl'] ?? '#',
                'display_url' => parse_url($savedAdData['destinationUrl'] ?? '', PHP_URL_HOST) ?: 'ai-itinc.com',
                'call_to_action' => $savedAdData['cta'] ?? 'Learn More',
                'ad_type' => !empty($savedAdData['productImage']) ? 'image' : 'text',
                'campaign_id' => $campaignAd['campaign_id'],
                'user_id' => $campaignAd['user_id']
            ];
            $adData = $savedAdData;
        }
    }
    
    if ($ad && !isset($adData)) {
        $adId = $ad['id'];
    }
}

if (!$ad) {
    http_response_code(404);
    die('Ad not found or not active');
}

// Track impression (only for acceleratedadz_ads, not campaign_ads)
try {
    if (isset($ad['ad_group_id'])) {
        // This is an acceleratedadz ad
        $date = date('Y-m-d');
        $impressionStmt = $db->prepare("
            INSERT INTO acceleratedadz_performance 
            (ad_id, ad_group_id, campaign_id, date, impressions, clicks, conversions, spend, revenue, 
             ctr, cpc, cpm, cpa, roas)
            VALUES (?, ?, ?, ?, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0)
            ON DUPLICATE KEY UPDATE
                impressions = impressions + 1,
                updated_at = NOW()
        ");
        $impressionStmt->execute([
            $ad['id'],
            $ad['ad_group_id'],
            $ad['campaign_id'],
            $date
        ]);
    }
    // For campaign_ads, we could track in ad_performance table if needed
} catch (PDOException $e) {
    error_log("AcceleratedAdz impression tracking error: " . $e->getMessage());
}

// Parse ad data if not already parsed
if (!isset($adData)) {
    $adData = json_decode($ad['ad_data'] ?? '{}', true);
}
$destinationUrl = $ad['destination_url'] ?? ($adData['destinationUrl'] ?? '#');
$displayUrl = $ad['display_url'] ?? parse_url($destinationUrl, PHP_URL_HOST) ?: 'ai-itinc.com';

// Add click tracking parameter
$trackingSource = isset($ad['ad_group_id']) ? 'acceleratedadz' : 'campaign';
$trackingUrl = url('acceleratedadz.php') . '?click=' . $adId . '&redirect=' . urlencode($destinationUrl) . '&source=' . $trackingSource;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($ad['headline'] ?? 'Advertisement') ?> - AcceleratedAdz</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
        }
        .ad-container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            max-width: 600px;
            width: 100%;
            overflow: hidden;
            transition: transform 0.3s ease;
        }
        .ad-container:hover {
            transform: translateY(-5px);
        }
        .ad-image {
            width: 100%;
            height: 300px;
            object-fit: cover;
            display: block;
        }
        .ad-content {
            padding: 30px;
        }
        .ad-headline {
            font-size: 28px;
            font-weight: bold;
            color: #333;
            margin-bottom: 15px;
            line-height: 1.3;
        }
        .ad-description {
            font-size: 16px;
            color: #666;
            margin-bottom: 20px;
            line-height: 1.6;
        }
        .ad-cta {
            display: inline-block;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px 40px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: bold;
            font-size: 18px;
            transition: opacity 0.3s ease;
            text-align: center;
            width: 100%;
        }
        .ad-cta:hover {
            opacity: 0.9;
        }
        .ad-footer {
            padding: 15px 30px;
            background: #f8f9fa;
            border-top: 1px solid #e9ecef;
            text-align: center;
            font-size: 12px;
            color: #999;
        }
        .ad-display-url {
            color: #0066cc;
            font-size: 14px;
            margin-bottom: 10px;
        }
        .video-container {
            position: relative;
            width: 100%;
            padding-bottom: 56.25%; /* 16:9 aspect ratio */
            height: 0;
            overflow: hidden;
        }
        .video-container iframe,
        .video-container video {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
        }
    </style>
</head>
<body>
    <div class="ad-container">
        <?php if (isset($ad['ad_group_id'])): ?>
            <!-- AcceleratedAdz native ad -->
            <?php if ($ad['ad_type'] === 'image' && !empty($ad['image_url'])): ?>
                <img src="<?= htmlspecialchars($ad['image_url']) ?>" alt="<?= htmlspecialchars($ad['headline'] ?? '') ?>" class="ad-image">
            <?php elseif ($ad['ad_type'] === 'video' && !empty($ad['video_url'])): ?>
                <div class="video-container">
                    <video controls autoplay muted>
                        <source src="<?= htmlspecialchars($ad['video_url']) ?>" type="video/mp4">
                    </video>
                </div>
            <?php elseif ($ad['ad_type'] === 'html5' && !empty($adData['html'])): ?>
                <div style="width: 100%; height: 300px;">
                    <?= $adData['html'] ?>
                </div>
            <?php endif; ?>
        <?php else: ?>
            <!-- Campaign ad (from saved_ads/campaign_ads) -->
            <?php if (!empty($ad['image_url']) || !empty($adData['productImage']) || !empty($adData['bgImage'])): ?>
                <div style="width: 100%; height: 300px; background: <?= htmlspecialchars($adData['bgColor'] ?? '#ffffff') ?>; background-image: <?= !empty($adData['bgImage']) ? 'url(' . htmlspecialchars($adData['bgImage']) . ')' : 'none' ?>; background-size: cover; background-position: center; display: flex; align-items: center; justify-content: center;">
                    <?php if (!empty($ad['image_url']) || !empty($adData['productImage'])): ?>
                        <img src="<?= htmlspecialchars($ad['image_url'] ?? $adData['productImage']) ?>" alt="<?= htmlspecialchars($ad['headline'] ?? '') ?>" style="max-width: 90%; max-height: 90%; object-fit: contain;">
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <div class="ad-content">
            <?php if (!empty($ad['headline'])): ?>
                <h1 class="ad-headline"><?= htmlspecialchars($ad['headline']) ?></h1>
            <?php endif; ?>
            
            <?php if (!empty($ad['description'])): ?>
                <p class="ad-description"><?= nl2br(htmlspecialchars($ad['description'])) ?></p>
            <?php endif; ?>
            
            <div class="ad-display-url"><?= htmlspecialchars($displayUrl) ?></div>
            
            <a href="<?= htmlspecialchars($trackingUrl) ?>" class="ad-cta" id="adClickLink">
                <?= htmlspecialchars($ad['call_to_action'] ?? 'Learn More') ?>
            </a>
        </div>
        
        <div class="ad-footer">
            Powered by <strong>AcceleratedAdz</strong> - AI-IT Inc
        </div>
    </div>

    <script>
        // Track click
        document.getElementById('adClickLink').addEventListener('click', function(e) {
            // Track click via API (non-blocking)
            const formData = new URLSearchParams();
            formData.append('action', 'track_click');
            formData.append('ad_id', '<?= $adId ?>');
            
            // Use sendBeacon for better reliability (before page unload)
            if (navigator.sendBeacon) {
                const blob = new Blob([formData.toString()], { type: 'application/x-www-form-urlencoded' });
                navigator.sendBeacon('<?= url('api/acceleratedadz.php') ?>', blob);
            }
            
            // Also use fetch for immediate tracking
            fetch('<?= url('api/acceleratedadz.php') ?>', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: formData.toString(),
                keepalive: true
            }).catch(err => console.error('Click tracking error:', err));
        });
    </script>
</body>
</html>

